-- Copyright 2024-2025 by Todd Hundersmarck (ThundR) 
-- All Rights Reserved

local thModName = g_currentModName
local thModPath = g_currentModDirectory
source(thModPath .. "scripts/THCore.lua")
THBetterWinch = {}
local THBetterWinch_mt = THUtils.createClass(THBetterWinch, THCore)
local debugFlagId = THDebugUtil.createFlagId("THBetterWinch", true)
THBetterWinch.debugFlagId = debugFlagId
THBetterWinch.DATA_KEY = "thBetterWinch"
THBetterWinch.XML_BASE_KEY = THBetterWinch.DATA_KEY
THBetterWinch.DEFAULT_ATTACH_RADIUS = 0.1
THBetterWinch.MIN_ATTACH_RADIUS = 0.01
THBetterWinch.MAX_ATTACH_RADIUS = 10
THBetterWinch.ATTACH_RADIUS_ACCEL_DELAY = 1000
function THBetterWinch.new(customMt)
    customMt = customMt or THBetterWinch_mt
    local self = THCore.new(THBetterWinch.DATA_KEY, thModName, customMt)
    if self ~= nil then
        self.objectsI3DFilename = "data/objects.i3d"
        self.attachableObjects = {}
        self.objectAttachNodes = {}
        self.objectAttachPoints = {}
        return self
    end
end
function THBetterWinch.onPreInit(self)
    local thSpecManager = self.thSpecManager
    thSpecManager:addSpecialization(THSpecType.VEHICLE, self.dataKey, "THVSpec_Winch", self.modName, "scripts/vehicles/specializations/THVSpec_Winch.lua")
    thSpecManager:addSpecialization(THSpecType.VEHICLE, self.dataKey .. "Object", "THVSpec_WinchObject", self.modName, "scripts/vehicles/specializations/THVSpec_WinchObject.lua")
end
function THBetterWinch.onLoadMapFinished(self, mission)
    self:load()
    local objectsI3DFilename = THUtils.getFilename(self.objectsI3DFilename, self.modPath)
    local objectsRootNode, sharedLoadRequestId = self.i3DManager:loadSharedI3DFile(objectsI3DFilename, false, false)
    local rootNode = getRootNode()
    local attachNode = getChildAt(objectsRootNode, 0)
    link(rootNode, attachNode)
    self.attachPointTemplate = attachNode
    self.sharedLoadRequestId = sharedLoadRequestId
end
function THBetterWinch.getObjectComponentIndex(self, objectId)
    local mission = self.mission
    if THUtils.argIsValid(THUtils.getIsType(objectId, THValueType.INTEGER) and objectId > 0, "objectId", objectId)
        and mission ~= nil
    then
        local object = mission:getNodeObject(objectId)
        if object ~= nil then
            if type(object.components) == THValueType.TABLE then
                for componentIndex, componentInfo in pairs(object.components) do
                    if I3DUtil.getIsLinkedToNode(componentInfo.node, objectId) then
                        return componentIndex
                    end
                end
                THDebugUtil.printMsg(debugFlagId, THDebugLevel.WARNING, "Specified object id [%s] is not a root node", objectId)
            end
            if type(object.nodeId) == THValueType.NUMBER and object.nodeId == objectId then
                return 1
            end
            if type(object.rootNode) == THValueType.NUMBER and object.rootNode == objectId then
                return 1
            end
            THDebugUtil.printMsg(debugFlagId, THDebugLevel.ERROR, "Component index not found")
        end
    else
        THDebugUtil.printMsg(debugFlagId, THDebugLevel.ERROR, THMessage.ARGUMENT_INVALID, "objectId", objectId)
    end
end
function THBetterWinch.getObjectComponentNode(self, object, componentIndex)
    if THUtils.argIsValid(THUtils.getIsType(object, Object), "object", object)
        and THUtils.argIsValid(THUtils.getIsType(componentIndex, THValueType.INTEGER), "componentIndex", componentIndex)
    then
        if type(object.components) == THValueType.TABLE then
            local componentInfo = object.components[componentIndex]
            if componentInfo ~= nil then
                if THUtils.getIsType(componentInfo.node, THValueType.INTEGER) and componentInfo.node > 0 then
                    return componentInfo.node
                end
                THDebugUtil.printMsg(debugFlagId, THDebugLevel.ERROR, "Invalid component [%s] node: %s", componentIndex, componentInfo.node)
            else
                THDebugUtil.printMsg(debugFlagId, THDebugLevel.ERROR, THMessage.ARGUMENT_INVALID, "componentIndex", componentIndex)
            end
        end
        if componentIndex == 1 then
            if THUtils.getIsType(object.nodeId, THValueType.INTEGER) and object.nodeId > 0 then
                return object.nodeId
            end
            if THUtils.getIsType(object.rootNode, THValueType.INTEGER) and object.rootNode > 0 then
                return object.rootNode
            end
            THDebugUtil.printMsg(debugFlagId, THDebugLevel.ERROR, "Could not find object component node")
        else
            THDebugUtil.printMsg(debugFlagId, THDebugLevel.ERROR, THMessage.ARGUMENT_INVALID, "componentIndex", componentIndex)
        end
    end
end
function THBetterWinch.injectObjectAttachPoint(self, object, parentNode, offsetX, offsetY, offsetZ, rotX, rotY, rotZ)
    if THUtils.argIsValid(THUtils.getIsType(object, Object), "object", object)
        and THUtils.argIsValid(THUtils.getIsType(parentNode, THValueType.INTEGER) and parentNode > 0, "parentNode", parentNode)
        and THUtils.argIsValid(type(offsetX) == THValueType.NUMBER, "offsetX", offsetX)
        and THUtils.argIsValid(type(offsetY) == THValueType.NUMBER, "offsetY", offsetY)
        and THUtils.argIsValid(type(offsetZ) == THValueType.NUMBER, "offsetZ", offsetZ)
        and THUtils.argIsValid(type(rotX) == THValueType.NUMBER, "rotX", rotX)
        and THUtils.argIsValid(type(rotY) == THValueType.NUMBER, "rotY", rotY)
        and THUtils.argIsValid(type(rotZ) == THValueType.NUMBER, "rotZ", rotZ)
    then
        if not entityExists(parentNode) then
            THUtils.errorMsg(nil, "Can not find attach point parent node: %s", parentNode)
        else
            local objectAttachPoints = self.objectAttachPoints[object]
            local isAttachPointFound = false
            if objectAttachPoints == nil then
                objectAttachPoints = {}
                self.objectAttachPoints[object] = objectAttachPoints
            else
                for _, otherAttachPointData in pairs(objectAttachPoints) do
                    if otherAttachPointData.parentNode == parentNode then
                        isAttachPointFound = true
                        break
                    end
                end
            end
            if not isAttachPointFound then
                local attachPointNode = clone(self.attachPointTemplate, false, false, false)
                if attachPointNode ~= nil and attachPointNode > 0 then
                    link(parentNode, attachPointNode)
                    setTranslation(attachPointNode, offsetX, offsetY, offsetZ)
                    setRotation(attachPointNode, rotX, rotY, rotZ)
                    local attachPointData = self:addObjectAttachPoint(object, attachPointNode)
                    if attachPointData ~= nil then
                        attachPointData.parentNode = parentNode
                        return attachPointData
                    end
                end
            end
        end
    end
end
function THBetterWinch.addObjectAttachPoint(self, object, nodeId)
    if THUtils.argIsValid(THUtils.getIsType(object, Object), "object", object)
        and THUtils.argIsValid(THUtils.getIsType(nodeId, THValueType.INTEGER) and nodeId > 0, "nodeId", nodeId)
    then
        if not entityExists(nodeId) then
            THUtils.errorMsg(nil, "Cannot find attach point node: %s", nodeId)
        else
            local objectAttachPoints = self.objectAttachPoints[object]
            local isAttachPointFound = false
            if objectAttachPoints == nil then
                objectAttachPoints = {}
                self.objectAttachPoints[object] = objectAttachPoints
            else
                for _, otherAttachPointData in pairs(objectAttachPoints) do
                    if otherAttachPointData.node == nodeId then
                        isAttachPointFound = true
                        break
                    end
                end
            end
            if not isAttachPointFound then
                local attachPointData = {
                    object = object,
                    node = nodeId
                }
                table.insert(objectAttachPoints, attachPointData)
                return attachPointData
            end
        end
    end
end
function THBetterWinch.getObjectAttachPoints(self, object)
    if object ~= nil then
        return self.objectAttachPoints[object]
    end
end
function THBetterWinch.validateAttachableObject(self, objectId)
    local mission = self.mission
    if type(objectId) ~= THValueType.NUMBER or objectId <= 0 then
        THDebugUtil.printMsg(debugFlagId, THDebugLevel.ERROR, THMessage.ARGUMENT_INVALID, "objectId", objectId)
    elseif not entityExists(objectId) then
        THDebugUtil.printMsg(debugFlagId, THDebugLevel.ERROR, "Object node does not exist")
    else
        local splitType = getSplitType(objectId)
        if type(splitType) == THValueType.NUMBER and splitType > 0 then
            THDebugUtil.printMsg(debugFlagId, THDebugLevel.ERROR, "Object node is a split type and not supported")
        elseif mission ~= nil then
            local object = mission.nodeToObject[objectId]
            if THUtils.getIsType(object, Object) then
                return true
            end
            THDebugUtil.printMsg(debugFlagId, THDebugLevel.ERROR, "No object associated with object node")
        else
            THDebugUtil.printMsg(debugFlagId, THDebugLevel.ERROR, "Mission table does not exist")
        end
    end
    return false
end
function THBetterWinch.addAttachableObject(self, objectId)
    local mission = self.mission
    if self:validateAttachableObject(objectId)
        and mission ~= nil
    then
        local object = mission:getNodeObject(objectId)
        if object == nil then
            return
        end
        local objectData = self:getAttachableObject(objectId)
        local componentIndex = self:getObjectComponentIndex(objectId)
        if objectData ~= nil then
            if componentIndex ~= objectData.component then
                THDebugUtil.printMsg(debugFlagId, THDebugLevel.ERROR, "Object [%s] component mismatch (%s >> %s)", objectId, componentIndex, objectData.component)
            end
        elseif componentIndex ~= nil and componentIndex > 0 then
            local objectSizeX = getUserAttribute(objectId, "thWidth")
            local objectSizeY = getUserAttribute(objectId, "thHeight")
            local objectSizeZ = getUserAttribute(objectId, "thLength")
            local objectRadius = getUserAttribute(objectId, "thRadius")
            if type(objectRadius) ~= THValueType.NUMBER or objectRadius <= 0 then
                objectRadius = THBetterWinch.DEFAULT_ATTACH_RADIUS
            end
            if type(objectSizeX) ~= THValueType.NUMBER or objectSizeX <= 0 then
                if type(object.size) == THValueType.TABLE
                    and type(object.size.width) == THValueType.NUMBER
                    and object.size.width > 0
                then
                    objectSizeX = object.size.width
                elseif type(object.width) == THValueType.NUMBER
                    and object.width > 0
                then
                    objectSizeX = object.width
                elseif type(object.diameter) == THValueType.NUMBER
                    and object.diameter > 0
                then
                    objectSizeX = object.diameter
                end
            end
            if type(objectSizeY) ~= THValueType.NUMBER or objectSizeY <= 0 then
                if type(object.size) == THValueType.TABLE
                    and type(object.size.height) == THValueType.NUMBER
                    and object.size.height > 0
                then
                    objectSizeY = object.size.height
                elseif type(object.height) == THValueType.NUMBER
                    and object.height > 0
                then
                    objectSizeY = object.height
                elseif type(object.diameter) == THValueType.NUMBER
                    and object.diameter > 0
                then
                    objectSizeY = object.diameter
                end
            end
            if type(objectSizeZ) ~= THValueType.NUMBER or objectSizeZ <= 0 then
                if type(object.size) == THValueType.TABLE
                    and type(object.size.length) == THValueType.NUMBER
                    and object.size.length > 0
                then
                    objectSizeZ = object.size.length
                elseif type(object.length) == THValueType.NUMBER
                    and object.length > 0
                then
                    objectSizeZ = object.length
                elseif type(object.diameter) == THValueType.NUMBER
                    and object.diameter > 0
                then
                    objectSizeZ = object.diameter
                end
            end
            if objectSizeX == nil then
                if objectSizeZ ~= nil then
                    if objectSizeY ~= nil then
                        objectSizeX = math.max(objectSizeY, objectSizeZ)
                    else
                        objectSizeX = objectSizeZ
                    end
                else
                    objectSizeZ = objectRadius * 2
                end
            end
            if objectSizeY == nil then
                if objectSizeZ ~= nil then
                    if objectSizeX ~= nil then
                        objectSizeY = math.max(objectSizeX, objectSizeZ)
                    else
                        objectSizeY = objectSizeZ
                    end
                else
                    objectSizeY = objectRadius * 2
                end
            end
            if objectSizeZ == nil then
                if objectSizeX ~= nil then
                    if objectSizeY ~= nil then
                        objectSizeZ = math.max(objectSizeX, objectSizeY)
                    else
                        objectSizeZ = objectSizeX
                    end
                else
                    objectSizeZ = objectRadius * 2
                end
            end
            objectRadius = math.max(objectSizeX, objectSizeZ) * 0.5
            objectData = {
                object = object,
                node = objectId,
                nodeName = getName(objectId),
                component = componentIndex,
                meshNodes = {
                    byId = {},
                    byIndex = {}
                },
                attachedObjects = {},
                size = { objectSizeX, objectSizeY, objectSizeZ },
                radius = objectRadius,
                isDirty = false
            }
            local function addMeshNode(pNodeId, pIsBaseNode)
                if type(pNodeId) == THValueType.NUMBER and pNodeId > 0
                    and objectData.meshNodes.byId[pNodeId] == nil
                    and entityExists(pNodeId) and getShapeIsCPUMesh(pNodeId)
                    and (pIsBaseNode == true or I3DUtil.getIsLinkedToNode(objectId, pNodeId))
                then
                    local meshData = {
                        parent = objectData,
                        node = pNodeId,
                        nodeName = getName(pNodeId),
                        size = { 1, 1, 1 },
                        index = #objectData.meshNodes.byIndex + 1,
                        upAxis = THAxis.Y,
                        canTarget = false
                    }
                    local meshSizeX = getUserAttribute(pNodeId, "thWidth")
                    local meshSizeY = getUserAttribute(pNodeId, "thHeight")
                    local meshSizeZ = getUserAttribute(pNodeId, "thLength")
                    local meshRadius = getUserAttribute(pNodeId, "thRadius")
                    if getHasCollision(pNodeId) then
                        meshData.canTarget = true
                    end
                    if type(meshRadius) ~= THValueType.NUMBER or meshRadius <= 0 then
                        meshRadius = THBetterWinch.DEFAULT_ATTACH_RADIUS
                    end
                    if type(meshSizeX) ~= THValueType.NUMBER or meshSizeX <= 0 then
                        if meshData.canTarget then
                            meshSizeX = meshRadius * 2
                        else
                            meshSizeX = objectSizeX
                        end
                    end
                    if type(meshSizeY) ~= THValueType.NUMBER or meshSizeY <= 0 then
                        if meshData.canTarget then
                            meshSizeY = meshRadius * 2
                        else
                            meshSizeY = objectSizeY
                        end
                    end
                    if type(meshSizeZ) ~= THValueType.NUMBER or meshSizeZ <= 0 then
                        if meshData.canTarget then
                            meshSizeZ = meshRadius * 2
                        else
                            meshSizeZ = objectSizeZ
                        end
                    end
                    meshRadius = math.max(meshSizeX, meshSizeZ) * 0.5
                    meshData.size = { meshSizeX, meshSizeY, meshSizeZ }
                    meshData.radius = meshRadius
                    objectData.meshNodes.byId[meshData.node] = meshData
                    objectData.meshNodes.byIndex[meshData.index] = meshData
                    return true
                end
                return false
            end
            addMeshNode(objectId, true)
            if object.getSupportsTensionBelts ~= nil
                and object:getSupportsTensionBelts()
                and object.getMeshNodes ~= nil
            then
                local meshNodes = object:getMeshNodes()
                if meshNodes ~= nil then
                    for nodeIndex = 1, #meshNodes do
                        addMeshNode(meshNodes[nodeIndex])
                    end
                end
            end
            local objectAttachPoints = self:getObjectAttachPoints(object)
            if objectAttachPoints ~= nil then
                for attachPointIndex = 1, #objectAttachPoints do
                    local attachPointData = objectAttachPoints[attachPointIndex]
                    addMeshNode(attachPointData.node)
                end
            end
            self.attachableObjects[objectData.node] = objectData
            if self.objectAttachNodes[objectData.object] == nil then
                self.objectAttachNodes[objectData.object] = {}
            end
            self.objectAttachNodes[objectData.object][objectData.component] = objectData
            if THDebugUtil.getIsEnabled(debugFlagId) then
                THUtils.displayMsg("Added object: %s", objectData.nodeName)
                THDebugUtil.printTable(objectData)
                THUtils.displayMsg("Attach nodes:")
                THDebugUtil.printTable(self.objectAttachNodes[objectData.object], 1)
                THUtils.displayMsg("Mesh nodes:")
                THDebugUtil.printTable(objectData.meshNodes, 2)
            end
        end
        return objectData
    end
end
function THBetterWinch.getAttachableObject(self, objectId)
    local mission = self.mission
    if mission ~= nil and objectId ~= nil then
        local objectData = self.attachableObjects[objectId]
        if objectData ~= nil then
            if objectData.node == objectId and entityExists(objectData.node)
                and objectData.object == mission:getNodeObject(objectData.node)
            then
                if objectData.currentMeshNode ~= nil then
                    if not entityExists(objectData.currentMeshNode) then
                        objectData.currentMeshNode = nil
                    end
                end
                return objectData
            end
            THDebugUtil.printMsg(debugFlagId, THDebugLevel.WARNING, "Object [%s] is no longer valid", objectData.node)
        end
    end
end
function THBetterWinch.getObjectAttachNodes(self, object)
    if object ~= nil then
        return self.objectAttachNodes[object]
    end
end
function THBetterWinch.getObjectClosestMeshNode(self, objectId, x, y, z, radius, force)
    if THUtils.argIsValid(radius == nil or (type(radius) == THValueType.NUMBER and radius > 0), "radius", radius)
        and THUtils.argIsValid(not force or force == true, "force", force)
        and THUtils.argIsValid(type(x) == THValueType.NUMBER, "x", x)
        and THUtils.argIsValid(type(y) == THValueType.NUMBER, "y", y)
        and THUtils.argIsValid(type(z) == THValueType.NUMBER, "z", z)
    then
        local objectData = self:getAttachableObject(objectId)
        if objectData ~= nil then
            local foundMeshNode = nil
            local minMeshDistance = radius
            local isObjectAttached = #objectData.attachedObjects > 0
            if objectData.currentMeshNode ~= nil then
                local meshData = objectData.meshNodes.byId[objectData.currentMeshNode]
                if meshData ~= nil then
                    foundMeshNode = meshData
                    local lx, ly, lz = worldToLocal(meshData.node, x, y, z)
                    local meshDistance = THUtils.getVector3Distance(lx, ly, lz)
                    if meshDistance ~= nil and meshDistance > 0 then
                        minMeshDistance = minMeshDistance
                    end
                else
                    objectData.currentMeshNode = nil
                end
            end
            if foundMeshNode == nil
                or (not isObjectAttached and not foundMeshNode.canTarget)
            then
                local searchAllNodes = foundMeshNode == nil and force == true
                for _, meshData in pairs(objectData.meshNodes.byIndex) do
                    if foundMeshNode == nil or meshData.node ~= foundMeshNode then
                        if not meshData.canTarget or searchAllNodes == true then
                            local lx, ly, lz = worldToLocal(meshData.node, x, y, z)
                            local meshDistance = THUtils.getVector3Distance(lx, ly, lz)
                            if meshDistance ~= nil
                                and (minMeshDistance == nil or meshDistance < minMeshDistance)
                            then
                                foundMeshNode = meshData
                                minMeshDistance = meshDistance
                            end
                        end
                    end
                end
            end
            if foundMeshNode ~= nil then
                if not isObjectAttached or objectData.currentMeshNode == nil then
                    objectData.currentMeshNode = foundMeshNode.node
                end
            end
            return foundMeshNode, minMeshDistance
        end
    end
end
function THBetterWinch.getAttachableObjectOffsetPosition(self, objectId, x, y, z, maxRadius, minLength)
    local meshData = self:getObjectClosestMeshNode(objectId, x, y, z)
    if meshData ~= nil then
        local lx, ly, lz = worldToLocal(meshData.node, x, y, z)
        local cx, cy, cz = nil, nil, nil
        local upX, upY, upZ = nil, nil, nil
        if meshData.upAxis == THAxis.X then
            cx, cy, cz = localToWorld(meshData.node, lx, 0, 0)
            upX, upY, upZ = localDirectionToWorld(meshData.node, 1, 0, 0)
        elseif meshData.upAxis == THAxis.Z then
            cx, cy, cz = localToWorld(meshData.node, 0, 0, lz)
            upX, upY, upZ = localDirectionToWorld(meshData.node, 0, 0, 1)
        else
            cx, cy, cz = localToWorld(meshData.node, 0, ly, 0)
            upX, upY, upZ = localDirectionToWorld(meshData.node, 0, 1, 0)
        end
        if cx ~= nil and upX ~= nil then
            local radius = meshData.radius
            if radius == nil or radius < THBetterWinch.MIN_ATTACH_RADIUS then
                if meshData.upAxis == THAxis.X then
                    radius = math.max(meshData.size[THAxis.Y], meshData.size[THAxis.Z]) * 0.5
                elseif meshData.upAxis == THAxis.Z then
                    radius = math.max(meshData.size[THAxis.X], meshData.size[THAxis.Y]) * 0.5
                else
                    radius = math.max(meshData.size[THAxis.X], meshData.size[THAxis.Z]) * 0.5
                end
                if radius == nil or radius < THBetterWinch.MIN_ATTACH_RADIUS then
                    radius = THBetterWinch.DEFAULT_ATTACH_RADIUS
                end
            end
            radius = THUtils.clamp(radius, THBetterWinch.MIN_ATTACH_RADIUS, THBetterWinch.MAX_ATTACH_RADIUS)
            if meshData.radius == nil then
                meshData.radius = radius
            end
            if maxRadius ~= nil and maxRadius > 0 then
                radius = math.min(radius, maxRadius)
            end
            if meshData.upAxis == THAxis.X or meshData.upAxis == THAxis.Z then
                if g_terrainNode ~= nil and g_terrainNode > 0 then
                    local ty = getTerrainHeightAtWorldPos(g_terrainNode, cx, cy, cz)
                    local meshHeight = meshData.size[THAxis.Y]
                    if cy - (meshHeight * 0.5) < ty then
                        cy = ty + (meshHeight * 0.5)
                    end
                end
            end
            if THDebugUtil.getIsEnabled(debugFlagId, THDebugLevel.UPDATE) then
                THUtils.displayMsg("Updating object offset position:")
                THUtils.displayMsg("- Radius: %s", radius)
                THUtils.displayMsg("- cx, cy, cz: %0.3f, %0.3f, %0.3f", cx, cy, cz)
                THUtils.displayMsg("- upX, upY, upZ: %0.3f, %0.3f, %0.3f", upX, upY, upZ)
                THUtils.displayMsg("")
            end
            return cx, cy, cz, upX, upY, upZ, radius
        end
    end
end
function THBetterWinch.setIsObjectAttached(self, objectId, winchObject, isAttached)
    if THUtils.argIsValid(type(objectId) == THValueType.NUMBER, "objectId", objectId)
        and THUtils.argIsValid(type(winchObject) == THValueType.TABLE, "winchObject", winchObject)
        and THUtils.argIsValid(not isAttached or isAttached == true, "isAttached", isAttached)
    then
        local objectData = self:getAttachableObject(objectId)
        if objectData ~= nil then
            if isAttached then
                local isFound = false
                for _, otherWinch in pairs(objectData.attachedObjects) do
                    if otherWinch == winchObject then
                        isFound = true
                        break
                    end
                end
                if not isFound then
                    table.insert(objectData.attachedObjects, winchObject)
                end
            elseif #objectData.attachedObjects > 0 then
                local winchIndex = 1
                while true do
                    if objectData.attachedObjects[winchIndex] == nil then
                        break
                    end
                    if objectData.attachedObjects[winchIndex] == winchObject then
                        table.remove(objectData.attachedObjects, winchIndex)
                    else
                        winchIndex = winchIndex + 1
                    end
                end
            end
        end
    end
end
function THBetterWinch.getIsObjectAttached(self, objectId)
    if type(objectId) == THValueType.TABLE then
        local objectAttachNodes = self:getObjectAttachNodes(objectId)
        if objectAttachNodes ~= nil then
            for _, objectData in pairs(objectAttachNodes) do
                if #objectData.attachedObjects > 0 then
                    return true
                end
            end
        end
    else
        local objectData = self:getAttachableObject(objectId)
        if objectData ~= nil then
            if #objectData.attachedObjects > 0 then
                return true
            end
        end
    end
    return false
end
function THBetterWinch.inj_readSplitShapeIdFromStream(self, superFunc, streamId, ...)
    if streamReadBool(streamId) then
        local object = NetworkUtil.readNodeObject(streamId)
        local componentIndex = streamReadUIntN(streamId, 16)
        local meshIndex, radius, upAxis = nil, nil, nil
        if streamReadBool(streamId) then
            meshIndex = streamReadUIntN(streamId, 16)
            radius = streamReadFloat32(streamId)
            upAxis = streamReadUIntN(streamId, 4)
        end
        local objectId = nil
        THUtils.call(function()
            objectId = self:getObjectComponentNode(object, componentIndex)
            if objectId == nil then
                objectId = 0
            else
                local objectData = self:addAttachableObject(objectId)
                if objectData ~= nil then
                    objectData.currentMeshNode = nil
                    if meshIndex ~= nil then
                        local meshData = objectData.meshNodes.byIndex[meshIndex]
                        if meshData ~= nil then
                            objectData.currentMeshNode = meshData.node
                            if radius ~= nil and radius > 0 then
                                meshData.radius = radius
                            end
                            if upAxis ~= nil and upAxis > 0 then
                                meshData.upAxis = upAxis
                            end
                        end
                    end
                    objectId = objectData.node
                else
                    objectId = 0
                end
            end
        end)
        if objectId == nil or objectId < 0 then
            objectId = 0
        end
        return objectId, 0, 0
    end
    return superFunc(streamId, ...)
end
function THBetterWinch.inj_writeSplitShapeIdToStream(self, superFunc, streamId, objectId, ...)
    local objectData = nil
    THUtils.call(function()
        objectData = self:getAttachableObject(objectId)
    end)
    if streamWriteBool(streamId, objectData ~= nil) then
        NetworkUtil.writeNodeObject(streamId, objectData.object)
        streamWriteUIntN(streamId, objectData.component, 16)
        local meshData = objectData.meshNodes.byId[objectData.currentMeshNode]
        if streamWriteBool(streamId, meshData ~= nil) then
            streamWriteUIntN(streamId, meshData.index, 16)
            streamWriteFloat32(streamId, meshData.radius or 0)
            streamWriteUIntN(streamId, meshData.upAxis or 0, 4)
        end
        return objectData.node, 0, 0
    end
    return superFunc(streamId, objectId, ...)
end
function THBetterWinch.inj_getTreeOffsetPosition(self, superFunc, objectId, x, y, z, maxRadius, minLength, ...)
    local objectData = nil
    THUtils.call(function()
        objectData = self:getAttachableObject(objectId)
    end)
    if objectData ~= nil then
        return self:getAttachableObjectOffsetPosition(objectId, x, y, z, maxRadius, minLength)
    end
    return superFunc(objectId, x, y, z, maxRadius, minLength, ...)
end
function THBetterWinch.inj_createSplitShapeTreeBelt(self, superFunc, beltData, objectId, tx, ty, tz, sx, sy, sz, upX, upY, upZ, hookOffset, ignoreYDirection, spacing, ...)
    if objectId ~= nil then
        THUtils.call(function()
            local meshData = self:getObjectClosestMeshNode(objectId, tx, ty, tz, nil, true)
            if meshData ~= nil then
                objectId = meshData.node
            end
        end)
    end
    return superFunc(beltData, objectId, tx, ty, tz, sx, sy, sz, upX, upY, upZ, hookOffset, ignoreYDirection, spacing, ...)
end
THUtils.call(function()
    local self = THBetterWinch.new()
    if self ~= nil then
        _G.g_thBetterWinch = self
        THUtils.setFunctionHook(g_thGlobalEnv, "readSplitShapeIdFromStream", false, false, self, THBetterWinch.inj_readSplitShapeIdFromStream)
        THUtils.setFunctionHook(g_thGlobalEnv, "writeSplitShapeIdToStream", false, false, self, THBetterWinch.inj_writeSplitShapeIdToStream)
        THUtils.setFunctionHook("SplitShapeUtil", "getTreeOffsetPosition", false, false, self, THBetterWinch.inj_getTreeOffsetPosition)
        THUtils.setFunctionHook("SplitShapeUtil", "createTreeBelt", false, false, self, THBetterWinch.inj_createSplitShapeTreeBelt)
    end
end)