-- Copyright 2024-2025 by Todd Hundersmarck (ThundR) 
-- All Rights Reserved

local thMain = g_thBetterWinch
local thI18n = thMain.i18n
local thInputManager = thMain.inputManager
THVSpec_Winch = {}
local debugFlagId = THDebugUtil.createFlagId("THVSpec_Winch", true)
THVSpec_Winch.debugFlagId = debugFlagId
THVSpec_Winch.MOD_NAME = thMain.modName
THVSpec_Winch.SPEC_NAME = thMain.dataKey
THVSpec_Winch.SPEC_TABLE_KEY = "spec_" .. THVSpec_Winch.MOD_NAME .. "." .. THVSpec_Winch.SPEC_NAME
THVSpec_Winch.COLLISION_MASK = CollisionFlag.DYNAMIC_OBJECT + CollisionFlag.VEHICLE + CollisionFlag.TREE
THVSpec_Winch.ACTION_TEXT = {
    ADJUST_ATTACH_RADIUS = thI18n:getText("thAction_adjustAttachRadius"),
    ROTATE_ATTACH_LOOP = thI18n:getText("thAction_rotateAttachLoop")
}
THVSpecData_Winch = {}
THWinchAttachTreeActivatable = {}
local THWinchAttachTreeActivatable_mt = THUtils.createClass(THWinchAttachTreeActivatable)
function THVSpec_Winch.prerequisitesPresent(specializations)
    if SpecializationUtil.hasSpecialization(Winch, specializations) then
        return true
    end
    return false
end
function THVSpec_Winch.registerEventListeners(vehicleType)
    SpecializationUtil.registerEventListener(vehicleType, "onPreLoad", THVSpec_Winch)
    SpecializationUtil.registerEventListener(vehicleType, "onLoad", THVSpec_Winch)
end
function Winch.registerOverwrittenFunctions(vehicleType)
    SpecializationUtil.registerOverwrittenFunction(vehicleType, "onWinchTreeRaycastCallback", THVSpec_Winch.onWinchTreeRaycastCallback)
    SpecializationUtil.registerOverwrittenFunction(vehicleType, "attachTreeToWinch", THVSpec_Winch.attachTreeToWinch)
    SpecializationUtil.registerOverwrittenFunction(vehicleType, "detachTreeFromWinch", THVSpec_Winch.detachTreeFromWinch)
end
function THVSpec_Winch.getSpecTable(self)
    if self ~= nil then
        return self[THVSpec_Winch.SPEC_TABLE_KEY]
    end
end
local getSpecTable = THVSpec_Winch.getSpecTable
function THVSpec_Winch.onPreLoad(self, savegame)
    THUtils.call(function()
        local specTable = getSpecTable(self)
        local winchSpec = self.spec_winch
        if specTable ~= nil and winchSpec ~= nil then
            specTable.vehicle = self
            specTable.objectsCache = {}
            specTable.delayedLoadObjects = {}
            specTable.attachedObjects = {}
            specTable.detachedObjects = {}
            THUtils.copyFunctions(specTable, THVSpecData_Winch)
        end
    end)
end
function THVSpec_Winch.onLoad(self, savegame)
    THUtils.call(function()
        local specTable = getSpecTable(self)
        local winchSpec = self.spec_winch
        if specTable ~= nil and winchSpec ~= nil then
            if winchSpec.isAttachable then
                if SpecializationUtil.hasSpecialization(Enterable, self.specializations) then
                    winchSpec.isAttachable = false
                end
            end
            local winchTexts = winchSpec.texts
            winchTexts.attachAnotherTree = thI18n:getText("thBetterWinch_attachAnotherObject")
            winchTexts.attachTree = thI18n:getText("thBetterWinch_attachObject")
            winchTexts.detachTree = thI18n:getText("thBetterWinch_detachObject")
            winchTexts.warningTooHeavy = thI18n:getText("thBetterWinch_objectTooHeavy")
            winchTexts.warningMaxNumTreesReached = thI18n:getText("thBetterWinch_maxObjectsReached")
        end
    end)
end
function THVSpec_Winch.onWinchTreeRaycastCallback(self, superFunc, objectId, wx, wy, wz, distance, dx, dy, dz, shapeIndex, hitShapeId, isLast, ...)
    local mission = thMain.mission
    local specTable = getSpecTable(self)
    local winchSpec = self.spec_winch
    local function appendFunc(...)
        if mission ~= nil and specTable ~= nil and winchSpec ~= nil
            and type(objectId) == THValueType.NUMBER and objectId > 0
            and winchSpec.treeRaycast ~= nil
        then
            THUtils.call(function()
                if isLast then
                    local raycastInfo = winchSpec.treeRaycast
                    local objectData = thMain:getAttachableObject(winchSpec.treeRaycast.lastValidTree)
                    local ownerFarmId = self:getActiveFarm()
                    local isDebugEnabled = THDebugUtil.getIsEnabled(debugFlagId, THDebugLevel.UPDATE)
                    if isDebugEnabled then
                        THUtils.displayMsg("Object id: %s [%s]", objectId, getName(objectId))
                        if type(hitShapeId) == THValueType.NUMBER and hitShapeId > 0 then
                            THUtils.displayMsg("Hit shape id: %s [%s]", hitShapeId, getName(hitShapeId))
                        end
                        THUtils.displayMsg("Shape index: %s", shapeIndex)
                        THUtils.displayMsg("")
                    end
                    if objectData ~= nil
                        or raycastInfo.lastValidTree == nil
                    then
                        local object = mission:getNodeObject(objectId)
                        local maxRadius = THBetterWinch.MAX_ATTACH_RADIUS
                        if object == nil or object == self then
                            raycastInfo.lastValidTree = nil
                            return
                        end
                        objectData = thMain:addAttachableObject(objectId)
                        if objectData == nil then
                            raycastInfo.lastValidTree = nil
                            return
                        end
                        local isObjectAttached = false
                        for ropeIndex = 1, #winchSpec.ropes do
                            local ropeInfo = winchSpec.ropes[ropeIndex]
                            for treeIndex = 1, #ropeInfo.attachedTrees do
                                if ropeInfo.attachedTrees[treeIndex].treeId == objectData.node then
                                    isObjectAttached = true
                                end
                            end
                        end
                        thMain:setIsObjectAttached(objectData.node, self, isObjectAttached == true)
                        if isObjectAttached then
                            raycastInfo.lastValidTree = nil
                            return
                        end
                        objectData.currentMeshNode = nil
                        if not mission.accessHandler:canFarmAccess(ownerFarmId, object) then
                            raycastInfo.lastValidTree = nil
                            return
                        end
                        if hitShapeId ~= nil and objectData.meshNodes.byId[hitShapeId] ~= nil then
                            objectData.currentMeshNode = hitShapeId
                        elseif objectData.meshNodes.byId[objectId] ~= nil then
                            objectData.currentMeshNode = objectId
                        end
                        local cx, cy, cz, upX, upY, upZ, radius = thMain:getAttachableObjectOffsetPosition(objectData.node, wx, wy, wz, maxRadius, 0.15)
                        if cx == nil then
                            objectData.currentMeshNode = nil
                            raycastInfo.lastValidTree = nil
                        else
                            if MathUtil.vector3Length(raycastInfo.startPos[1] - wx, raycastInfo.startPos[2] - wy, raycastInfo.startPos[3] - wz) <= raycastInfo.maxDistance then
                                raycastInfo.lastValidTree = objectData.node
                                raycastInfo.lastInValidTree = nil
                            else
                                raycastInfo.lastValidTree = nil
                                raycastInfo.lastInValidTree = objectData.node
                            end
                            raycastInfo.treeTargetPos[1] = wx
                            raycastInfo.treeTargetPos[2] = wy
                            raycastInfo.treeTargetPos[3] = wz
                            raycastInfo.treeCenterPos[1] = cx
                            raycastInfo.treeCenterPos[2] = cy
                            raycastInfo.treeCenterPos[3] = cz
                            raycastInfo.treeUp[1] = upX
                            raycastInfo.treeUp[2] = upY
                            raycastInfo.treeUp[3] = upZ
                            raycastInfo.treeRadius = radius
                            if isDebugEnabled then
                                THUtils.displayMsg("Raycast information:")
                                THDebugUtil.printTable(raycastInfo, 1)
                                local meshData = objectData.meshNodes.byId[objectData.currentMeshNode]
                                if meshData ~= nil then
                                    THUtils.displayMsg("Current mesh node:")
                                    THDebugUtil.printTable(meshData)
                                else
                                    THUtils.errorMsg(false, "Cannot find object %q attach point data", objectData.nodeName)
                                end
                            end
                            self:raiseActive()
                        end
                    end
                end
            end)
        end
        return ...
    end
    return appendFunc(superFunc(self, objectId, wx, wy, wz, distance, dx, dy, dz, shapeIndex, hitShapeId, isLast, ...))
end
function THVSpec_Winch.attachTreeToWinch(self, superFunc, objectId, x, y, z, ropeIndex, ...)
    local specTable = getSpecTable(self)
    local winchSpec = self.spec_winch
    local isObjectAttached = false
    local objectData = nil
    local attachedTrees = nil
    if specTable ~= nil and winchSpec ~= nil and objectId ~= nil and ropeIndex ~= nil then
        THUtils.call(function()
            local ropeInfo = winchSpec.ropes[ropeIndex]
            objectData = thMain:getAttachableObject(objectId)
            if objectData ~= nil
                and ropeInfo ~= nil and ropeInfo.attachedTrees ~= nil
            then
                attachedTrees = ropeInfo.attachedTrees
                for _, attachInfo in pairs(attachedTrees) do
                    if attachInfo.treeId == objectData.node then
                        isObjectAttached = true
                        break
                    end
                end
            end
        end)
    end
    local function appendFunc(...)
        if not isObjectAttached
            and specTable ~= nil and winchSpec ~= nil
            and objectData ~= nil and attachedTrees ~= nil
        then
            THUtils.call(function()
                for _, attachInfo in pairs(attachedTrees) do
                    if attachInfo.treeId == objectData.node then
                        isObjectAttached = true
                        break
                    end
                end
                specTable:setAttachableObjectActive(objectData.node, isObjectAttached == true, true)
            end)
        end
        return ...
    end
    return appendFunc(superFunc(self, objectId, x, y, z, ropeIndex, ...))
end
function THVSpec_Winch.detachTreeFromWinch(self, superFunc, ropeIndex, ...)
    local specTable = getSpecTable(self)
    local winchSpec = self.spec_winch
    local attachedTrees, detachedObjects = nil, nil
    if specTable ~= nil and winchSpec ~= nil and ropeIndex ~= nil then
        THUtils.call(function()
            detachedObjects = THUtils.clearTable(specTable.detachedObjects)
            local ropeInfo = winchSpec.ropes[ropeIndex]
            if ropeInfo ~= nil and ropeInfo.attachedTrees ~= nil then
                attachedTrees = ropeInfo.attachedTrees
                for _, attachInfo in pairs(attachedTrees) do
                    local objectData = thMain:getAttachableObject(attachInfo.treeId)
                    if objectData ~= nil then
                        detachedObjects[objectData.node] = true
                    end
                end
            end
        end)
    end
    local function appendFunc(...)
        if specTable ~= nil and winchSpec ~= nil
            and attachedTrees ~= nil and detachedObjects ~= nil
        then
            THUtils.call(function()
                for _, attachInfo in pairs(attachedTrees) do
                    if detachedObjects[attachInfo.treeId] then
                        specTable:setAttachableObjectActive(attachInfo.treeId, true)
                        detachedObjects[attachInfo.treeId] = nil
                    end
                end
                for objectId in pairs(detachedObjects) do
                    specTable:setAttachableObjectActive(objectId, false, true)
                    detachedObjects[objectId] = nil
                end
            end)
        end
        return ...
    end
    return appendFunc(superFunc(self, ropeIndex, ...))
end
function THVSpec_Winch.onWinchTreeShapeMounted(self, superFunc, objectId, otherVehicle, ...)
    local specTable = getSpecTable(self)
    local winchSpec = self.spec_winch
    local prependSuccess = false
    local oldRopesTable = nil
    if specTable ~= nil and winchSpec ~= nil and otherVehicle ~= self then
        THUtils.call(function()
            oldRopesTable = winchSpec.ropes
            winchSpec.ropes = {}
            prependSuccess = true
        end)
    end
    local function appendFunc(...)
        if prependSuccess and winchSpec ~= nil then
            winchSpec.ropes = oldRopesTable
        end
        return ...
    end
    return appendFunc(superFunc(self, objectId, otherVehicle, ...))
end
function THVSpec_Winch.inj_registerXMLPaths(superFunc, xmlSchema, xmlPath, ...)
    local function appendFunc(...)
        if xmlSchema ~= nil and xmlPath ~= nil then
            THUtils.call(function()
                local xmlSchemaSavegame = Vehicle.xmlSchemaSavegame
                if xmlSchemaSavegame ~= nil then
                    local xmlSubPath = "vehicles.vehicle(?)." .. thMain.xmlKey
                    local ropePath = xmlSubPath .. ".rope(?)"
                    THUtils.registerXMLPath(xmlSchemaSavegame, XMLValueType.INT, ropePath, "#index", "Rope internal index")
                    local objectPath = ropePath .. ".attachedObject(?)"
                    THUtils.registerXMLPath(xmlSchemaSavegame, XMLValueType.VECTOR_TRANS, objectPath, "#translation", "Hook current position")
                    THUtils.registerXMLPath(xmlSchemaSavegame, XMLValueType.STRING, objectPath, "#uniqueId", "Object unique id")
                    THUtils.registerXMLPath(xmlSchemaSavegame, XMLValueType.INT, objectPath, "#component", "Object component index", 1)
                    THUtils.registerXMLPath(xmlSchemaSavegame, XMLValueType.INT, objectPath, "#meshIndex", "Object attach mesh index", 0)
                    THUtils.registerXMLPath(xmlSchemaSavegame, XMLValueType.FLOAT, objectPath, "#radius", "Object attach radius")
                    THUtils.registerXMLPath(xmlSchemaSavegame, XMLValueType.INT, objectPath, "#upAxis", "Attach ring orientation")
                    ForestryPhysicsRope.registerSavegameXMLPaths(xmlSchemaSavegame, objectPath .. ".physicsRope")
                end
            end)
        end
        return ...
    end
    return appendFunc(superFunc(xmlSchema, xmlPath, ...))
end
THUtils.setFunctionHook("Winch", "registerXMLPaths", false, false, nil, THVSpec_Winch.inj_registerXMLPaths)
function THVSpec_Winch.inj_onLoadFinished(self, superFunc, savegame, ...)
    local specTable = getSpecTable(self)
    local winchSpec = self.spec_winch
    local function appendFunc(...)
        if specTable ~= nil and winchSpec ~= nil
            and savegame ~= nil and savegame.xmlFile ~= nil
            and not savegame.resetVehicles
        then
            THUtils.call(function()
                local xmlFile = savegame.xmlFile
                local xmlSubKey = savegame.key .. "." .. thMain.xmlKey
                xmlFile:iterate(xmlSubKey .. ".rope", function(_, pRopeKey)
                    local ropeIndex = THUtils.getXMLValue(xmlFile, XMLValueType.INT, pRopeKey, "#index", 1)
                    if ropeIndex >= 1 then
                        xmlFile:iterate(pRopeKey .. ".attachedObject", function(_, pObjectKey)
                            local translation = THUtils.getXMLValue(xmlFile, nil, pObjectKey, "#translation", nil, true)
                            if translation ~= nil and #translation == 3 then
                                local uniqueId = THUtils.getXMLValue(xmlFile, nil, pObjectKey, "#uniqueId")
                                local componentIndex = THUtils.getXMLValue(xmlFile, nil, pObjectKey, "#component", 1)
                                local meshIndex = THUtils.getXMLValue(xmlFile, nil, pObjectKey, "#meshIndex", 0)
                                local radius = THUtils.getXMLValue(xmlFile, nil, pObjectKey, "#radius")
                                local upAxis = THUtils.getXMLValue(xmlFile, nil, pObjectKey, "#upAxis")
                                local positionData = ForestryPhysicsRope.loadPositionDataFromSavegame(xmlFile, pObjectKey .. ".physicsRope")
                                if uniqueId ~= nil and uniqueId ~= ""
                                    and positionData ~= nil
                                then
                                    local delayedLoadData = {
                                        uniqueId = uniqueId,
                                        ropeIndex = ropeIndex,
                                        componentIndex = componentIndex,
                                        meshIndex = meshIndex,
                                        radius = radius,
                                        upAxis = upAxis,
                                        positionData = positionData,
                                        translation = translation
                                    }
                                    table.insert(specTable.delayedLoadObjects, delayedLoadData)
                                end
                            end
                        end)
                    end
                end)
            end)
        end
        return ...
    end
    return appendFunc(superFunc(self, savegame, ...))
end
THUtils.setFunctionHook("Winch", "onLoadFinished", false, true, nil, THVSpec_Winch.inj_onLoadFinished)
function THVSpec_Winch.inj_saveToXMLFile(self, superFunc, xmlFile, xmlKey, ...)
    local specTable = getSpecTable(self)
    local winchSpec = self.spec_winch
    local hasCachedTrees = false
    if specTable ~= nil and winchSpec ~= nil
        and xmlFile ~= nil and xmlKey ~= nil
    then
        THUtils.call(function()
            if winchSpec.ropes ~= nil then
                local objectsCache = THUtils.clearTable(specTable.objectsCache)
                local xmlSubKey = string.sub(xmlKey, 1, -7) .. "." .. thMain.xmlKey
                local ropeXMLIndex = 0
                for ropeIndex = 1, #winchSpec.ropes do
                    local ropeInfo = winchSpec.ropes[ropeIndex]
                    local ropeKey = string.format("%s.rope(%d)", xmlSubKey, ropeXMLIndex)
                    THUtils.setXMLValue(xmlFile, nil, ropeKey, "#index", ropeIndex)
                    objectsCache[ropeIndex] = {
                        attachedTrees = ropeInfo.attachedTrees
                    }
                    local cachedTrees = objectsCache[ropeIndex].attachedTrees
                    if cachedTrees ~= nil then
                        ropeInfo.attachedTrees = {}
                        for attachIndex = 1, #cachedTrees do
                            local attachInfo = cachedTrees[attachIndex]
                            local objectData = thMain:getAttachableObject(attachInfo.treeId)
                            if objectData == nil then
                                table.insert(ropeInfo.attachedTrees, attachInfo)
                            elseif objectData.object ~= nil and objectData.object.getUniqueId ~= nil then
                                local object = objectData.object
                                local objectKey = string.format("%s.attachedObject(%d)", ropeKey, attachIndex - 1)
                                local uniqueId = object:getUniqueId()
                                if uniqueId ~= nil and uniqueId ~= "" then
                                    local componentIndex = objectData.component or 1
                                    local meshData = objectData.meshNodes.byId[objectData.currentMeshNode]
                                    THUtils.setXMLValue(xmlFile, nil, objectKey, "#uniqueId", uniqueId)
                                    THUtils.setXMLValue(xmlFile, nil, objectKey, "#component", componentIndex)
                                    if meshData ~= nil then
                                        THUtils.setXMLValue(xmlFile, nil, objectKey, "#meshIndex", meshData.index)
                                        if meshData.radius ~= nil then
                                            THUtils.setXMLValue(xmlFile, nil, objectKey, "#radius", meshData.radius)
                                        end
                                        if meshData.upAxis ~= nil then
                                            THUtils.setXMLValue(xmlFile, nil, objectKey, "#upAxis", meshData.upAxis)
                                        end
                                    end
                                    local wx, wy, wz = getWorldTranslation(attachInfo.activeHookData.hookId)
                                    THUtils.setXMLValue(xmlFile, nil, objectKey, "#translation", wx, wy, wz)
                                    if ropeInfo.mainRope ~= nil then
                                        ropeInfo.mainRope:saveToXMLFile(xmlFile, objectKey .. ".physicsRope")
                                    end
                                end
                            end
                        end
                        hasCachedTrees = true
                    end
                    ropeXMLIndex = ropeXMLIndex + 1
                end
            end
        end)
    end
    local function appendFunc(...)
        if specTable ~= nil and winchSpec ~= nil then
            if hasCachedTrees then
                THUtils.call(function()
                    for ropeIndex = #specTable.objectsCache, 1, -1 do
                        local cacheData = specTable.objectsCache[ropeIndex]
                        local ropeInfo = winchSpec.ropes[ropeIndex]
                        if ropeInfo ~= nil then
                            ropeInfo.attachedTrees = cacheData.attachedTrees
                        end
                        specTable.objectsCache[ropeIndex] = nil
                    end
                end)
            end
        end
        return ...
    end
    return appendFunc(superFunc(self, xmlFile, xmlKey, ...))
end
THUtils.setFunctionHook("Winch", "saveToXMLFile", false, true, nil, THVSpec_Winch.inj_saveToXMLFile)
function THVSpec_Winch.inj_onPostUpdate(self, superFunc, dt, ...)
    local mission = thMain.mission
    local specTable = getSpecTable(self)
    local winchSpec = self.spec_winch
    local oldTreeColFlag = CollisionFlag.TREE
    if specTable ~= nil and winchSpec ~= nil then
        THUtils.call(function()
            CollisionFlag.TREE = THVSpec_Winch.COLLISION_MASK
            if mission ~= nil then
                for delayedIndex = #specTable.delayedLoadObjects, 1, -1 do
                    local delayedLoadData = specTable.delayedLoadObjects[delayedIndex]
                    local object = mission:getObjectByUniqueId(delayedLoadData.uniqueId)
                    if object ~= nil then
                        local objectId = thMain:getObjectComponentNode(object, delayedLoadData.componentIndex or 1)
                        if objectId ~= nil and objectId > 0 and entityExists(objectId) then
                            local objectData = thMain:addAttachableObject(objectId)
                            if objectData ~= nil then
                                if delayedLoadData.meshIndex ~= nil and delayedLoadData.meshIndex > 0 then
                                    local meshData = objectData.meshNodes.byIndex[delayedLoadData.meshIndex]
                                    if meshData ~= nil then
                                        objectData.currentMeshNode = meshData.node
                                        if delayedLoadData.radius ~= nil then
                                            meshData.radius = delayedLoadData.radius
                                        end
                                        if delayedLoadData.upAxis ~= nil then
                                            meshData.upAxis = delayedLoadData.upAxis
                                        end
                                    end
                                end
                                local wx, wy, wz = THUtils.unpack(delayedLoadData.translation)
                                self:attachTreeToWinch(objectData.node, wx, wy, wz, delayedLoadData.ropeIndex, delayedLoadData.positionData)
                            end
                        end
                    end
                    specTable.delayedLoadObjects[delayedIndex] = nil
                end
            end
        end)
    end
    local function appendFunc(...)
        CollisionFlag.TREE = oldTreeColFlag
        if specTable ~= nil and winchSpec ~= nil and winchSpec.ropes ~= nil then
            THUtils.call(function()
                for ropeIndex = 1, #winchSpec.ropes do
                    local ropeInfo = winchSpec.ropes[ropeIndex]
                    if ropeInfo.attachedTrees ~= nil then
                        for _, attachInfo in pairs(ropeInfo.attachedTrees) do
                            if attachInfo.treeId ~= nil then
                                local objectData = thMain:getAttachableObject(attachInfo.treeId)
                                if objectData ~= nil then
                                    specTable:setAttachableObjectActive(objectData.node, true)
                                end
                            end
                        end
                    end
                end
            end)
        end
        return ...
    end
    return appendFunc(superFunc(self, dt, ...))
end
THUtils.setFunctionHook("Winch", "onPostUpdate", false, true, nil, THVSpec_Winch.inj_onPostUpdate)
function THVSpecData_Winch.setAttachableObjectActive(specTable, objectId, isActive, force)
    local self = specTable.vehicle
    if THUtils.argIsValid(not isActive or isActive == true, "isActive", isActive)
        and THUtils.argIsValid(not force or force == true, "force", force)
        and objectId ~= nil
    then
        isActive = THUtils.getNoNil(isActive, true)
        local objectData = thMain:getAttachableObject(objectId)
        if objectData ~= nil then
            local object = objectData.object
            local isObjectDirty = objectData.isDirty or force == true
            thMain:setIsObjectAttached(objectData.node, self, isActive == true)
            if type(object.isa) == THValueType.FUNCTION then
                if object:isa(Vehicle) then
                    local rootVehicle = object
                    if object.getRootVehicle ~= nil then
                        rootVehicle = object:getRootVehicle() or object
                    end
                    local isControlled = rootVehicle ~= nil
                        and rootVehicle.getIsControlled ~= nil
                        and rootVehicle:getIsControlled() == true
                    local function activateVehicle(pVehicle)
                        local attachableSpec = pVehicle.spec_attachable
                        local wheelSpec = pVehicle.spec_wheels
                        local isAttached = false
                        if isObjectDirty then
                            if attachableSpec ~= nil then
                                isAttached = pVehicle.getAttacherVehicle ~= nil
                                    and pVehicle:getAttacherVehicle() ~= nil
                                if not isAttached then
                                    if pVehicle.getIsSupportAnimationAllowed ~= nil
                                        and attachableSpec.supportAnimations ~= nil
                                    then
                                        local animSpeed = -1
                                        if not isActive then
                                            animSpeed = 1
                                        end
                                        for _, animation in pairs(attachableSpec.supportAnimations) do
                                            if pVehicle:getIsSupportAnimationAllowed(animation) then
                                                pVehicle:playAnimation(animation.animationName, animSpeed, nil, true)
                                            end
                                        end
                                    end
                                    if wheelSpec ~= nil and wheelSpec.wheels ~= nil then
                                        for _, wheel in pairs(wheelSpec.wheels) do
                                            if wheel.wheelChocks ~= nil then
                                                for _, chock in pairs(wheel.wheelChocks) do
                                                    chock:update(isActive == true)
                                                end
                                            end
                                        end
                                    end
                                end
                            end
                        end
                        if not isAttached and not isControlled then
                            if isObjectDirty then
                                if pVehicle.brake ~= nil then
                                    local brakeForce = 1
                                    if isActive then
                                        brakeForce = 0.4
                                    else
                                        if pVehicle.getBrakeForce ~= nil then
                                            brakeForce = pVehicle:getBrakeForce()
                                            if brakeForce == nil or brakeForce <= 0 then
                                                brakeForce = 1
                                            end
                                        end
                                    end
                                    pVehicle:brake(brakeForce)
                                end
                            end
                            if isActive then
                                pVehicle:raiseActive()
                            end
                        end
                    end
                    if rootVehicle ~= nil then
                        activateVehicle(rootVehicle)
                        if rootVehicle.getChildVehicles ~= nil then
                            local childVehicles = rootVehicle:getChildVehicles()
                            if childVehicles ~= nil then
                                for _, childVehicle in pairs(childVehicles) do
                                    activateVehicle(childVehicle)
                                end
                            end
                        end
                    else
                        activateVehicle(object)
                    end
                end
            end
            if objectData.isDirty then
                if THDebugUtil.getIsEnabled(debugFlagId) then
                    THUtils.displayMsg("Updating attachment status: %s (%s)", objectData.nodeName, isActive == true)
                    THDebugUtil.printTable(objectData)
                end
                objectData.isDirty = false
            end
        end
    end
end
function THVSpecData_Winch.adjustWinchAttachRadius(specTable, adjustValue)
    local self = specTable.vehicle
    local winchSpec = self.spec_winch
    if winchSpec ~= nil and winchSpec.treeRaycast ~= nil then
        if type(adjustValue) == THValueType.NUMBER and adjustValue ~= 0 then
            local objectData = thMain:getAttachableObject(winchSpec.treeRaycast.lastValidTree)
            if objectData ~= nil and objectData.currentMeshNode ~= nil then
                local meshData = objectData.meshNodes.byId[objectData.currentMeshNode]
                if meshData ~= nil then
                    meshData.radius = THUtils.clamp(meshData.radius + adjustValue, THBetterWinch.MIN_ATTACH_RADIUS, THBetterWinch.MAX_ATTACH_RADIUS)
                end
            end
        end
    end
end
function THWinchAttachTreeActivatable.getCustomData(self)
    local customData = THUtils.getDataTable(self, THWinchAttachTreeActivatable)
    return customData
end
function THWinchAttachTreeActivatable.resetActionEvents(self, inputContext)
    for eventName, eventId in pairs(self.actionEventIds) do
        thInputManager:removeActionEvent(eventId)
        self.actionEventIds[eventName] = nil
    end
end
function THWinchAttachTreeActivatable.updateActionEvents(self, inputContext)
    local vehicle = self.parent.vehicle
    local winchSpec = nil
    if vehicle ~= nil then
        winchSpec = vehicle.spec_winch
    end
    if winchSpec ~= nil
        and winchSpec.treeRaycast ~= nil
    then
        for _, eventId in pairs(self.actionEventIds) do
            local isEventActive = true
            if eventId == self.actionEventIds.adjustAttachRadius
                or eventId == self.actionEventIds.rotateAttachLoop
            then
                if winchSpec.treeRaycast.lastValidTree ~= nil then
                    local objectData = thMain:getAttachableObject(winchSpec.treeRaycast.lastValidTree)
                    if objectData == nil or objectData.currentMeshNode == nil
                        or objectData.meshNodes.byId[objectData.currentMeshNode] == nil
                    then
                        isEventActive = false
                    end
                else
                    isEventActive = false
                end
            end
            thInputManager:setActionEventTextVisibility(eventId, isEventActive)
            thInputManager:setActionEventActive(eventId, isEventActive)
        end
    else
        for _, eventId in pairs(self.actionEventIds) do
            thInputManager:setActionEventTextVisibility(eventId, false)
            thInputManager:setActionEventActive(eventId, false)
        end
    end
end
function THWinchAttachTreeActivatable.onActionAdjustAttachRadius(self, actionName, inputValue, callbackState, isAnalog)
    THUtils.call(function()
        local vehicle = self.parent.vehicle
        local specTable = getSpecTable(vehicle)
        if specTable ~= nil then
            if inputValue ~= self.lastAttachRadiusDirection then
                self.lastAttachRadiusTimer = 0
                self.lastAttachRadiusDirection = inputValue
                self.attachRadiusAccelTimer = 0
                self.attachRadiusAccelFactor = 1
            else
                self.lastAttachRadiusTimer = 500
            end
            if inputValue ~= 0 then
                local adjustValue = 0.001 * inputValue * self.attachRadiusAccelFactor
                specTable:adjustWinchAttachRadius(adjustValue)
            end
        end
    end)
end
function THWinchAttachTreeActivatable.onActionRotateAttachLoop(self, actionName, inputValue, callbackState, isAnalog)
    THUtils.call(function()
        local vehicle = self.parent.vehicle
        local specTable = getSpecTable(vehicle)
        local winchSpec = vehicle.spec_winch
        if specTable ~= nil and winchSpec ~= nil
            and winchSpec.treeRaycast ~= nil
            and winchSpec.treeRaycast.lastValidTree ~= nil
        then
            local objectData = thMain:getAttachableObject(winchSpec.treeRaycast.lastValidTree)
            if objectData ~= nil and objectData.currentMeshNode ~= nil then
                local meshData = objectData.meshNodes.byId[objectData.currentMeshNode]
                if meshData ~= nil then
                    local upAxis = meshData.upAxis + 1
                    if upAxis > THAxis.Z then
                        upAxis = THAxis.X
                    end
                    if upAxis == THAxis.X then
                        meshData.radius = math.max(meshData.size[THAxis.Y], meshData.size[THAxis.Z]) * 0.5
                    elseif upAxis == THAxis.Z then
                        meshData.radius = math.max(meshData.size[THAxis.X], meshData.size[THAxis.Y]) * 0.5
                    else
                        meshData.radius = math.max(meshData.size[THAxis.X], meshData.size[THAxis.Z]) * 0.5
                    end
                    meshData.upAxis = upAxis
                end
            end
        end
    end)
end
function THWinchAttachTreeActivatable.inj_registerCustomInput(self, superFunc, activatable, inputContext, ...)
    local function appendFunc(...)
        THUtils.call(function()
            self:resetActionEvents(inputContext)
            local _, eventId = thInputManager:registerActionEvent(InputAction.TH_BETTER_WINCH_ADJUST_RADIUS, self, THWinchAttachTreeActivatable.onActionAdjustAttachRadius, false, false, true, true)
            if eventId ~= nil then
                thInputManager:setActionEventTextPriority(eventId, GS_PRIO_VERY_HIGH)
                thInputManager:setActionEventText(eventId, THVSpec_Winch.ACTION_TEXT.ADJUST_ATTACH_RADIUS)
            end
            self.actionEventIds.adjustAttachRadius = eventId
            _, eventId = thInputManager:registerActionEvent(InputAction.TH_BETTER_WINCH_ROTATE_LOOP, self, THWinchAttachTreeActivatable.onActionRotateAttachLoop, false, true, false, true)
            if eventId ~= nil then
                thInputManager:setActionEventTextPriority(eventId, GS_PRIO_VERY_HIGH)
                thInputManager:setActionEventText(eventId, THVSpec_Winch.ACTION_TEXT.ROTATE_ATTACH_LOOP)
            end
            self.actionEventIds.rotateAttachLoop = eventId
            self:updateActionEvents(inputContext)
        end)
        return ...
    end
    return appendFunc(superFunc(activatable, inputContext, ...))
end
function THWinchAttachTreeActivatable.inj_removeCustomInput(self, superFunc, activatable, inputContext, ...)
    local function appendFunc(...)
        THUtils.call(function()
            self:resetActionEvents(inputContext)
        end)
        return ...
    end
    return appendFunc(superFunc(activatable, inputContext, ...))
end
function THWinchAttachTreeActivatable.inj_update(self, superFunc, activatable, dt, ...)
    local function appendFunc(...)
        THUtils.call(function()
            self:updateActionEvents()
            if self.lastAttachRadiusTimer > 0 then
                if self.attachRadiusAccelTimer >= THBetterWinch.ATTACH_RADIUS_ACCEL_DELAY then
                    self.attachRadiusAccelFactor = math.max(1, self.attachRadiusAccelFactor + (0.01 * dt))
                else
                    self.attachRadiusAccelTimer = self.attachRadiusAccelTimer + dt
                end
                self.lastAttachRadiusTimer = self.lastAttachRadiusTimer - dt
            else
                self.attachRadiusAccelTimer = 0
                self.attachRadiusAccelFactor = 1
            end
        end)
        return ...
    end
    return appendFunc(superFunc(activatable, dt, ...))
end
function THWinchAttachTreeActivatable.inj_new(superFunc, vehicle, rope, ...)
    local function appendFunc(rActivatable, ...)
        if rActivatable ~= nil then
            THUtils.call(function()
                local self = THUtils.createDataTable(rActivatable, THWinchAttachTreeActivatable, THWinchAttachTreeActivatable_mt)
                if self ~= nil then
                    self.actionEventIds = {}
                    self.lastAttachRadiusTimer = 0
                    self.lastAttachRadiusDirection = 0
                    self.attachRadiusAccelTimer = 0
                    self.attachRadiusAccelFactor = 1
                    THUtils.setFunctionHook(rActivatable, "registerCustomInput", false, false, self, THWinchAttachTreeActivatable.inj_registerCustomInput)
                    THUtils.setFunctionHook(rActivatable, "removeCustomInput", false, false, self, THWinchAttachTreeActivatable.inj_removeCustomInput)
                    THUtils.setFunctionHook(rActivatable, "update", false, false, self, THWinchAttachTreeActivatable.inj_update)
                end
            end)
        end
        return rActivatable, ...
    end
    return appendFunc(superFunc(vehicle, rope, ...))
end
THUtils.setFunctionHook("WinchAttachTreeActivatable", "new", false, false, nil, THWinchAttachTreeActivatable.inj_new)