-- Copyright 2024 by Todd Hundersmarck (ThundR)
-- All Rights Reserved

_G.THDebugLevel = {
    CRITICAL = 1,
    ERROR = 2,
    WARNING = 3,
    NORMAL = 4,
    UPDATE = 5,
    LOOP = 6
}
_G.THDebugLevelId = {}
for id, index in pairs(THDebugLevel) do
    THDebugLevelId[index] = id
end
THSetDebugEnabledEvent = {}
local THSetDebugEnabledEvent_mt = Class(THSetDebugEnabledEvent, Event)
THSetDebugLevelEvent = {}
local THSetDebugLevelEvent_mt = Class(THSetDebugLevelEvent, Event)
THUtils.initClass(THSetDebugEnabledEvent, "THSetDebugEnabledEvent", THClassType.EVENT)
THUtils.initClass(THSetDebugLevelEvent, "THSetDebugLevelEvent", THClassType.EVENT)
THDebugUtil = {}
THDebugUtil.debugFlagIds = {}
THDebugUtil.globalDebugFlag = false
THDebugUtil.currentDebugLevel = THDebugLevel.NORMAL
function THDebugUtil.getFlagId(flagId)
    if flagId == nil then
        flagId = ""
    elseif type(flagId) == THValueType.STRING then
        if flagId ~= "" then
            flagId = flagId:upper()
        end
    else
        THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "flagId", flagId)
        return
    end
    if flagId == "" or THDebugUtil.debugFlagIds[flagId] ~= nil then
        return flagId
    end
    THUtils.errorMsg(nil, "Debug flag %q does not exist")
end
function THDebugUtil.getFlagIds()
    return THDebugUtil.debugFlagIds
end
function THDebugUtil.getDebugLevel()
    return THDebugUtil.currentDebugLevel
end
function THDebugUtil.getDebugLevelById(debugLevelId)
    local idType = type(debugLevelId)
    local debugLevel = nil
    if idType == THValueType.STRING then
        debugLevel = THDebugLevel[debugLevelId:upper()]
        if debugLevel == nil then
            debugLevelId = THUtils.toNumber(debugLevelId, true)
            if debugLevelId ~= nil then
                debugLevel = THDebugLevelId[debugLevelId]
            end
        end
    elseif idType == THValueType.NUMBER then
        debugLevel = THDebugLevelId[debugLevelId]
    else
        THUtils.errorMsg(nil, "Invalid debug level: %q", debugLevelId)
    end
    return debugLevel
end
function THDebugUtil.setDebugLevel(debugLevelId)
    local debugLevel = THDebugUtil.getDebugLevelById(debugLevelId)
    local success = false
    if debugLevel ~= nil then
        THDebugUtil.currentDebugLevel = debugLevel
        success = true
    end
    return success, THDebugUtil.currentDebugLevel
end
function THDebugUtil.createFlagId(flagName)
    if THUtils.argIsValid(THUtils.validateId(flagName), "flagName", flagName) then
        local flagId = flagName:upper()
        if THDebugUtil.debugFlagIds[flagId] == nil then
            THDebugUtil.debugFlagIds[flagId] = false
        end
        return flagId
    end
end
function THDebugUtil.getIsEnabled(flagName, debugLevelId)
    local reqDebugLevel = nil
    if debugLevelId == nil then
        reqDebugLevel = THDebugUtil.currentDebugLevel
    else
        reqDebugLevel = THDebugUtil.getDebugLevelById(debugLevelId)
    end
    local flagId = THDebugUtil.getFlagId(flagName)
    local isDebugEnabled = false
    if flagId ~= nil
        and reqDebugLevel ~= nil and reqDebugLevel <= THDebugUtil.currentDebugLevel
    then
        if flagId == "" then
            isDebugEnabled = THDebugUtil.globalDebugFlag == true
        else
            isDebugEnabled = THDebugUtil.debugFlagIds[flagId] == true
        end
    end
    return isDebugEnabled, flagId
end
function THDebugUtil.setIsEnabled(flagName, isEnabled)
    local flagId = THDebugUtil.getFlagId(flagName)
    if flagId ~= nil then
        isEnabled = THUtils.getNoNil(isEnabled, true)
        if THUtils.argIsValid(not isEnabled or isEnabled == true, "isEnabled", isEnabled) then
            if flagId == "" then
                THDebugUtil.globalDebugFlag = isEnabled == true
                return true, THDebugUtil.globalDebugFlag, flagId
            else
                THDebugUtil.debugFlagIds[flagId] = isEnabled == true
                return true, THDebugUtil.debugFlagIds[flagId], flagId
            end
            local enabledText = "enabled"
            if isEnabled ~= true then
                enabledText = "disabled"
            end
            if flagId == "" then
                THUtils.displayMsg("Debugger is %s", enabledText)
            else
                THUtils.displayMsg("Debugging for %q is %s", flagId, enabledText)
            end
            return true, isEnabled == true, flagId
        end
    end
    return false
end
function THDebugUtil.resetDebug()
    THDebugUtil.setIsEnabled(nil, false)
    THDebugUtil.setDebugLevel(THDebugLevel.NORMAL)
end
function THDebugUtil.printMsg(flagName, debugLevelId, text, ...)
    if THDebugUtil.getIsEnabled(flagName, debugLevelId) then
        local debugLevel = THDebugUtil.getDebugLevelById(debugLevelId)
        if debugLevel == THDebugLevel.CRITICAL then
            THUtils.errorMsg(true, text, ...)
        elseif debugLevel == THDebugLevel.ERROR then
            THUtils.errorMsg(nil, text, ...)
        elseif debugLevel == THDebugLevel.WARNING then
            THUtils.errorMsg(false, text, ...)
        else
            THUtils.displayMsg(text, ...)
        end
    end
end
function THDebugUtil.printTable(target, ...)
    if g_thDebugTools ~= nil then
        return g_thDebugTools:printTable(target, ...)
    end
    local targetName = target
    if type(target) == THValueType.STRING then
        target = THUtils.getTableValue(nil, targetName)
    end
    if THUtils.argIsValid(type(target) == THValueType.TABLE, "target", targetName) then
        THUtils.displayMsg("Printing %s")
        for key, value in pairs(target) do
            THUtils.displayMsg("[%s]: %s", key, value)
        end
    end
end
THDebugUtil.dumpTable = THDebugUtil.printTable
function THDebugUtil.traceFunction(...)
    if g_thDebugTools ~= nil then
        return g_thDebugTools:traceFunction(...)
    end
end
function THSetDebugEnabledEvent.emptyNew()
    return Event.new(THSetDebugEnabledEvent_mt)
end
function THSetDebugEnabledEvent.new(flagName, isEnabled)
    local self = THSetDebugEnabledEvent.emptyNew()
    self.isEventValid = false
    if THUtils.argIsValid(not isEnabled or isEnabled == true, "isEnabled", isEnabled) then
        local flagId = THDebugUtil.getFlagId(flagName)
        if flagId ~= nil then
            if g_server ~= nil then
                THUtils.errorMsg(true, THMessage.EVENT_NO_SERVER)
            elseif not THUtils.getIsMasterUser() then
                THUtils.errorMsg(false, THMessage.EVENT_ADMIN_ONLY)
            else
                self.flagId = flagId
                self.isEnabled = THUtils.getNoNil(isEnabled, true)
                self.isEventValid = true
            end
        end
    end
    return self
end
function THSetDebugEnabledEvent.readStream(self, streamId, connection)
    if not connection:getIsServer() then
        if streamReadBool(streamId) then
            self.flagId = streamReadString(streamId)
            self.isEnabled = streamReadBool(streamId)
            self:run(connection)
        end
    else
        THUtils.errorMsg(true, THMessage.EVENT_NO_CLIENT_READ)
    end
end
function THSetDebugEnabledEvent.writeStream(self, streamId, connection)
    if connection:getIsServer() then
        if streamWriteBool(streamId, self.isEventValid) then
            streamWriteString(streamId, self.flagId)
            streamWriteBool(streamId, self.isEnabled)
        end
    else
        THUtils.errorMsg(true, THMessage.EVENT_NO_SERVER_WRITE)
    end
end
function THSetDebugEnabledEvent.run(self, connection)
    if not connection:getIsServer() then
        if self.isEventValid then
            THDebugUtil.setIsEnabled(self.flagId, self.isEnabled)
        end
    end
end
function THSetDebugEnabledEvent.sendEvent(flagName, isEnabled)
    local success = false
    if g_server == nil and g_client ~= nil then
        local newEvent = THSetDebugEnabledEvent.new(flagName, isEnabled)
        if newEvent ~= nil and newEvent.isEventValid then
            g_client:getServerConnection():sendEvent(newEvent)
            success = true
        end
    end
    return success
end
function THSetDebugLevelEvent.emptyNew()
    return Event.new(THSetDebugLevelEvent_mt)
end
function THSetDebugLevelEvent.new(debugLevelId)
    local self = THSetDebugLevelEvent.emptyNew()
    local debugLevel = THDebugUtil.getDebugLevelById(debugLevelId)
    self.isEventValid = false
    if THUtils.argIsValid(debugLevel ~= nil, "debugLevelId", debugLevelId) then
        if g_server ~= nil then
            THUtils.errorMsg(true, THMessage.EVENT_NO_SERVER)
        elseif not THUtils.getIsMasterUser() then
            THUtils.errorMsg(false, THMessage.EVENT_ADMIN_ONLY)
        else
            self.debugLevel = math.floor(debugLevel)
            self.isEventValid = true
        end
    end
    return self
end
function THSetDebugLevelEvent.readStream(self, streamId, connection)
    if not connection:getIsServer() then
        if streamReadBool(streamId) then
            self.debugLevel = streamReadUIntN(streamId, 4)
            self:run(connection)
        end
    else
        THUtils.errorMsg(true, THMessage.EVENT_NO_CLIENT_READ)
    end
end
function THSetDebugLevelEvent.writeStream(self, streamId, connection)
    if connection:getIsServer() then
        if streamWriteBool(streamId, self.isEventValid) then
            streamWriteUIntN(streamId, self.debugLevel, 4)
        end
    else
        THUtils.errorMsg(true, THMessage.EVENT_NO_SERVER_WRITE)
    end
end
function THSetDebugLevelEvent.run(self, connection)
    if not connection:getIsServer() then
        if self.isEventValid then
            THDebugUtil.setDebugLevel(self.debugLevel)
        end
    end
end
function THSetDebugLevelEvent.sendEvent(debugLevelId)
    local success = false
    if g_server == nil and g_client ~= nil then
        local newEvent = THSetDebugLevelEvent.new(debugLevelId)
        if newEvent ~= nil and newEvent.isEventValid then
            g_client:getServerConnection():sendEvent(newEvent)
            success = true
        end
    end
    return success
end