-- Copyright 2024 by Todd Hundersmarck (ThundR)
-- All Rights Reserved

local thModName = g_currentModName
local thModPath = g_currentModDirectory
source(thModPath .. "scripts/utilities/THUtils.lua")
source(thModPath .. "scripts/utilities/THDebugUtil.lua")
source(thModPath .. "scripts/managers/THModManager.lua")
source(thModPath .. "scripts/managers/THSpecManager.lua")
local configTypeBits = g_thGlobalEnv.ConfigurationUtil.SEND_NUM_BITS or 7
if configTypeBits < 10 then
    g_thGlobalEnv.ConfigurationUtil.SEND_NUM_BITS = 10
end
THDesignKit = {}
local THDesignKit_mt = THUtils.createClass(THDesignKit)
THDesignKit.MOD_NAME = thModName
THDesignKit.MOD_PATH = thModPath
THDesignKit.DATA_KEY = "thDesignKit"
THDesignKit.XML_KEY = THDesignKit.DATA_KEY
THDesignKit.OLD_XML_KEY = "thPlaceableDesign"
THDesignKit.NUM_CONFIG_SLOTS = 8
THDesignKit.OBJECT_TYPE = {
    VEHICLE = 1,
    PLACEABLE = 2
}
THDesignKit.CONFIG_TYPE = {
    DESIGN = 1,
    COLOR = 2
}
THDesignKit.OBJECT_CHANGE_TYPE = {
    VISIBILITY = "visibility",
    TRANSLATION = "translation",
    ROTATION = "rotation",
    SCALE = "scale",
    CENTER_OF_MASS = "centerOfMass"
}
function THDesignKit.new(customMt)
    customMt = customMt or THDesignKit_mt
    if THUtils.argIsValid(type(customMt) == THValueType.TABLE, "customMt", customMt) then
        local self = setmetatable({}, customMt)
        self.isServer = g_server ~= nil
        self.isClient = g_client ~= nil
        self.modName = thModName
        self.modPath = thModPath
        self.thModManager = g_thModManager
        self.thSpecManager = g_thSpecManager
        self.inputManager = g_inputBinding
        self.i18n = g_i18n
        self.dataKey = THDesignKit.DATA_KEY
        self.xmlKey = THDesignKit.XML_KEY
        self.configXMLKeys = {
            THDesignKit.OLD_XML_KEY,
            self.xmlKey
        }
        self.objectChangeNodes = {}
        self.isInitialized = false
        self.objectTypes = {
            byId = {},
            byIndex = {}
        }
        for objectTypeId, objectTypeIndex in pairs(THDesignKit.OBJECT_TYPE) do
            local objectTypeData = {
                id = objectTypeId,
                name = objectTypeId:lower(),
                index = objectTypeIndex
            }
            if objectTypeIndex == THDesignKit.OBJECT_TYPE.VEHICLE then
                objectTypeData.manager = g_vehicleConfigurationManager
            elseif objectTypeIndex == THDesignKit.OBJECT_TYPE.PLACEABLE then
                objectTypeData.manager = g_placeableConfigurationManager
            end
            self.objectTypes.byId[objectTypeData.id] = objectTypeData
            self.objectTypes.byIndex[objectTypeData.index] = objectTypeData
            local configTypes = {
                byId = {},
                byIndex = {}
            }
            for configTypeId, configTypeIndex in pairs(THDesignKit.CONFIG_TYPE) do
                local configTypeData = {
                    id = configTypeId,
                    name = configTypeId:lower(),
                    index = configTypeIndex,
                }
                local configTypeNameProper = THUtils.properCase(configTypeData.name)
                configTypeData.configName = string.format("th%s", configTypeNameProper)
                configTypeData.configTitle = self.i18n:getText("thConfigTitle_" .. configTypeData.name)
                if objectTypeIndex == THDesignKit.OBJECT_TYPE.PLACEABLE then
                    if configTypeIndex == THDesignKit.CONFIG_TYPE.COLOR then
                        configTypeData.typeClass = PlaceableConfigurationItemColor
                    else
                        configTypeData.typeClass = PlaceableConfigurationItem
                    end
                end
                configTypes.byId[configTypeData.id] = configTypeData
                configTypes.byIndex[configTypeData.index] = configTypeData
            end
            objectTypeData.configTypeKey = string.format("%sConfigurations", objectTypeData.name)
            objectTypeData.configTypes = configTypes
        end
        return self
    end
end
function THDesignKit.getObjectTypeData(self, objectTypeId, verbose)
    local objectTypeIdType = type(objectTypeId)
    local objectTypeData = nil
    verbose = THUtils.validateArg(not verbose or verbose == true, "verbose", verbose, true)
    if objectTypeIdType == THValueType.STRING then
        objectTypeData = self.objectTypes.byId[objectTypeId:upper()]
    elseif objectTypeIdType == THValueType.NUMBER then
        objectTypeData = self.objectTypes.byIndex[objectTypeId]
    elseif objectTypeId ~= nil then
        verbose = true
    end
    if objectTypeData == nil and verbose == true then
        THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "objectTypeId", objectTypeId)
    end
    return objectTypeData
end
function THDesignKit.getConfigTypeData(self, configTypeId, objectTypeId, verbose)
    local objectTypeData = self:getObjectTypeData(objectTypeId, verbose)
    if objectTypeData ~= nil then
        local configTypeIdType = type(configTypeId)
        local configTypeData = nil
        if configTypeIdType == THValueType.STRING then
            configTypeData = objectTypeData.configTypes.byId[configTypeId:upper()]
        elseif configTypeIdType == THValueType.NUMBER then
            configTypeData = objectTypeData.configTypes.byIndex[configTypeId]
        elseif configTypeId ~= nil then
            verbose = true
        end
        if configTypeData == nil and verbose == true then
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "configTypeId", configTypeId)
        end
        return configTypeData
    end
end
function THDesignKit.loadConfigsFromXML(self, objectTypeId, xmlFile, xmlKey, modName, modPath, isOldKey)
    local objectTypeData = self:getObjectTypeData(objectTypeId, true)
    local modData = self.thModManager:getLoadedMod(modName, true)
    local success = false
    if objectTypeData ~= nil and modData ~= nil
        and THUtils.argIsValid(type(xmlFile) == THValueType.TABLE, "xmlFile", xmlFile)
        and THUtils.argIsValid(type(xmlKey) == THValueType.STRING, "xmlKey", xmlKey)
        and THUtils.argIsValid(not isOldKey or isOldKey == true, "isOldKey", isOldKey)
    then
        if THUtils.hasXMLProperty(xmlFile, xmlKey) then
            success = true
            xmlFile:iterate(xmlKey .. ".config", function(_, pConfigKey)
                local configName = xmlFile:getString(pConfigKey .. "#name")
                local configTitle = xmlFile:getString(pConfigKey .. "#title")
                local configTypeId = xmlFile:getString(pConfigKey .. "#type")
                local configTypeData = nil
                if configTypeId ~= nil then
                    configTypeData = self:getConfigTypeData(configTypeId, objectTypeData.index)
                end
                local configPathText = string.format("%s: %s", modData.name, pConfigKey)
                if configTypeData == nil then
                    THUtils.xmlErrorMsg(configPathText, nil, THMessage.INVALID_VALUE, "type", configTypeId)
                elseif not THUtils.validateId(configName) then
                    THUtils.xmlErrorMsg(configPathText, nil, THMessage.INVALID_VALUE, "name", configName)
                else
                    local configManager = objectTypeData.manager
                    local configXMLKey = self.xmlKey
                    if isOldKey then
                        configXMLKey = THDesignKit.OLD_XML_KEY
                    end
                    if configManager ~= nil then
                        local configInfo = configManager:getConfigurationDescByName(configName)
                        if configInfo == nil then
                            if configTitle == nil then
                                configTitle = configTypeData.configTitle
                            else
                                configTitle = self.i18n:convertText(configTitle, modData.name)
                            end
                            configManager:addConfigurationType(configName, configTitle, configXMLKey, configTypeData.typeClass)
                        end
                    end
                end
            end)
        end
    end
    return success
end
function THDesignKit.loadConfigurations(self)
    for _, objectTypeData in ipairs(self.objectTypes.byIndex) do
        local configManager = objectTypeData.manager
        if configManager ~= nil then
            if objectTypeData.index == THDesignKit.OBJECT_TYPE.PLACEABLE then
                for _, configTypeData in ipairs(objectTypeData.configTypes.byIndex) do
                    if configTypeData.typeClass ~= nil then
                        for slotNum = 1, THDesignKit.NUM_CONFIG_SLOTS do
                            local configName = string.format("%s%d", configTypeData.configName, slotNum)
                            configManager:addConfigurationType(configName, configTypeData.configTitle, self.xmlKey, configTypeData.typeClass)
                        end
                    end
                end
            end
        end
    end
    local modsArray, numMods = self.thModManager:getLoadedMods()
    if numMods > 0 then
        for _, modData in ipairs(modsArray) do
            local modXMLFile = THUtils.loadXMLFile("THModDesc", modData.file, nil, true)
            if modXMLFile ~= nil then
                local configsKey = string.format("modDesc.%s.configurations", THDesignKit.OLD_XML_KEY)
                if THUtils.hasXMLProperty(modXMLFile, configsKey) then
                    self:loadConfigsFromXML(THDesignKit.OBJECT_TYPE.PLACEABLE, modXMLFile, configsKey, modData.name, modData.path, true)
                else
                    for _, objectTypeData in pairs(self.objectTypes.byIndex) do
                        configsKey = string.format("modDesc.%s.%s", self.xmlKey, objectTypeData.configTypeKey)
                        self:loadConfigsFromXML(objectTypeData.index, modXMLFile, configsKey, modData.name, modData.path)
                    end
                end
                THUtils.deleteXMLFile(modXMLFile)
            end
        end
    end
    return true
end
function THDesignKit.updateConfigurations(self, placeable, configs, configData, boughtConfigs, isPurchased, purchasePrice, farmId, noEventSend)
    isPurchased = THUtils.validateArg(not isPurchased or isPurchased == true, "isPurchased", isPurchased, false)
    purchasePrice = purchasePrice or 0
    local configManager = g_placeableConfigurationManager
    local mission = g_currentMission
    local success = false
    if THUtils.argIsValid(THUtils.getIsType(placeable, Placeable), "placeable", placeable)
        and THUtils.argIsValid(configs == nil or type(configs) == THValueType.TABLE, "configs", configs)
        and THUtils.argIsValid(configData == nil or type(configData) == THValueType.TABLE, "configData", configData)
        and THUtils.argIsValid(boughtConfigs == nil or type(boughtConfigs) == THValueType.TABLE, "boughtConfigs", boughtConfigs)
        and THUtils.argIsValid(type(purchasePrice) == THValueType.NUMBER and purchasePrice >= 0, "purchasePrice", purchasePrice)
    then
        success = true
        if mission ~= nil
            and (self.isServer or noEventSend == true)
        then
            configs = configs or placeable.configurations
            configData = configData or placeable.configurationData
            if boughtConfigs == nil then
                placeable:setConfigurations(configs, placeable.boughtConfigurations or {}, configData)
            else
                placeable:setConfigurations(configs, boughtConfigs, configData)
            end
            if isPurchased then
                if self.isServer then
                    THUtils.clearTable(placeable.boughtConfigurations)
                    for configName, configId in pairs(placeable.configurations) do
                        ConfigurationUtil.addBoughtConfiguration(configManager, placeable, configName, configId)
                    end
                    boughtConfigs = placeable.boughtConfigurations
                    farmId = farmId or placeable:getOwnerFarmId()
                    if purchasePrice > 0 and farmId ~= nil
                        and farmId ~= AccessHandler.EVERYONE
                        and farmId ~= AccessHandler.NOBODY
                        and farmId ~= FarmManager.SPECTATOR_FARM_ID
                    then
                        mission:addMoney(-purchasePrice, farmId, MoneyType.PROPERTY_MAINTENANCE, true, true)
                    end
                end
            end
            ConfigurationUtil.raiseConfigurationItemEvent(placeable, "onLoad")
            ConfigurationUtil.raiseConfigurationItemEvent(placeable, "onPostLoad")
            ConfigurationUtil.raiseConfigurationItemEvent(placeable, "onLoadFinished")
        end
        if noEventSend ~= true then
            success = THPlaceableChangeConfigEvent.sendEvent(placeable, configs, configData, boughtConfigs, isPurchased, purchasePrice, farmId)
        end
    end
    return success
end
function THDesignKit.setObjectChangeDefaults(self, object, apply)
    if THUtils.argIsValid(type(object) == THValueType.TABLE, "object", object)
        and THUtils.argIsValid(not apply or apply == true, "apply", apply)
    then
        local nodeId = object.node
        if THUtils.getIsType(object.parent, Placeable)
            and THUtils.getNoNil(nodeId, 0) > 0
            and object.values ~= nil and next(object.values) ~= nil
        then
            local xmlFilename = object.parent.configFileName
            local nodePath = I3DUtil.getNodePath(nodeId)
            if nodePath ~= nil and xmlFilename ~= nil
                and self.objectChangeNodes[xmlFilename] ~= nil
                and self.objectChangeNodes[xmlFilename][nodePath] ~= nil
            then
                local nodeData = self.objectChangeNodes[xmlFilename][nodePath]
                for _, valueDesc in pairs(object.values) do
                    local valueName = valueDesc.name
                    if valueName ~= nil and valueName ~= ""
                        and (valueDesc.active == nil or valueDesc.inactive == nil)
                    then
                        local srcValues = THUtils.getNoNil(valueDesc.active, valueDesc.inactive)
                        local defaultData = nodeData.defaultValues[valueName]
                        if defaultData == nil then
                            defaultData = {
                                isActive = false
                            }
                            nodeData.defaultValues[valueName] = defaultData
                            local srcType = type(srcValues)
                            local tgtValues = nil
                            if srcType == THValueType.TABLE then
                                tgtValues = THUtils.copyTable(srcValues)
                            else
                                tgtValues = { srcValues }
                            end
                            if next(tgtValues) ~= nil then
                                if valueName == THDesignKit.OBJECT_CHANGE_TYPE.VISIBILITY then
                                    local isVisible = getVisibility(nodeId)
                                    if isVisible ~= nil then
                                        tgtValues[1] = isVisible
                                        defaultData.isActive = true
                                    end
                                elseif valueName == THDesignKit.OBJECT_CHANGE_TYPE.TRANSLATION
                                    or valueName == THDesignKit.OBJECT_CHANGE_TYPE.ROTATION
                                    or valueName == THDesignKit.OBJECT_CHANGE_TYPE.SCALE
                                    or valueName == THDesignKit.OBJECT_CHANGE_TYPE.CENTER_OF_MASS
                                then
                                    local x, y, z = nil, nil, nil
                                    if valueName == THDesignKit.OBJECT_CHANGE_TYPE.TRANSLATION then
                                        x, y, z = getTranslation(nodeId)
                                    elseif valueName == THDesignKit.OBJECT_CHANGE_TYPE.ROTATION then
                                        x, y, z = getRotation(nodeId)
                                    elseif valueName == THDesignKit.OBJECT_CHANGE_TYPE.SCALE then
                                        x, y, z = getScale(nodeId)
                                    elseif valueName == THDesignKit.OBJECT_CHANGE_TYPE.CENTER_OF_MASS then
                                        x, y, z = getCenterOfMass(nodeId)
                                    end
                                    if x ~= nil and y ~= nil and z ~= nil then
                                        tgtValues[1] = x
                                        tgtValues[2] = y
                                        tgtValues[3] = z
                                        defaultData.isActive = true
                                    end
                                end
                            end
                            if defaultData.isActive then
                                if srcType == THValueType.TABLE then
                                    defaultData.values = tgtValues
                                else
                                    defaultData.values = tgtValues[1]
                                end
                            end
                        end
                        if apply == true and defaultData.isActive then
                            if valueDesc.active == nil then
                                valueDesc.active = defaultData.values
                            elseif valueDesc.inactive == nil then
                                valueDesc.inactive = defaultData.values
                            end
                        end
                    end
                end
            end
        end
    end
end
function THDesignKit.inj_finalizeTypes(self, superFunc, manager, ...)
    if not self.isInitialized then
        THUtils.call(function()
            self:loadConfigurations()
        end)
        self.isInitialized = true
    end
    return superFunc(manager, ...)
end
function THDesignKit.inj_setShaderParameter(self, superFunc, nodeId, paramName, r, g, b, a, ...)
    if self.configNodes ~= nil then
        THUtils.call(function()
            if THUtils.getNoNil(nodeId, 0) > 0 then
                local nodePath = I3DUtil.getNodePath(nodeId)
                local nodeData = self.configNodes[nodePath]
                if nodeData ~= nil then
                    if nodeData.shaderMapping ~= nil then
                        local newParamName = nodeData.shaderMapping[paramName]
                        if newParamName ~= nil then
                            paramName = newParamName
                        end
                    end
                end
            end
        end)
    end
    return superFunc(nodeId, paramName, r, g, b, a, ...)
end
function THDesignKit.inj_registerPlaceableColorXMLPaths(self, superFunc, xmlSchema, xmlRootPath, xmlPath, ...)
    local function appendFunc(...)
        if xmlSchema ~= nil and xmlRootPath ~= nil then
            THUtils.call(function()
                for _, configXMLKey in ipairs(self.configXMLKeys) do
                    local xmlSubPath = xmlRootPath .. "." .. configXMLKey
                    local nodeParamPath = xmlSubPath .. ".nodeParams.node(?)"
                    THUtils.registerXMLPath(xmlSchema, XMLValueType.NODE_INDEX, nodeParamPath .. "#node", "Node id")
                    THUtils.registerXMLPath(xmlSchema, XMLValueType.STRING, nodeParamPath .. "#shader", "Shader parameter")
                end
            end)
        end
        return ...
    end
    return appendFunc(superFunc(xmlSchema, xmlRootPath, xmlPath, ...))
end
function THDesignKit.inj_onPostLoadPlaceableColor(self, superFunc, item, placeable, ...)
    if placeable ~= nil then
        THUtils.call(function()
            local xmlFile = placeable.xmlFile
            local components = placeable.components
            local i3dMappings = placeable.i3dMappings
            local objectTypeData = self:getObjectTypeData(THDesignKit.OBJECT_TYPE.PLACEABLE)
            if xmlFile ~= nil and objectTypeData ~= nil then
                local configManager = objectTypeData.manager
                local configInfo = configManager:getConfigurationDescByName(item.configName)
                if configInfo ~= nil then
                    local itemData = THUtils.createDataTable(item, self)
                    if itemData ~= nil then
                        if itemData.configNodes == nil then
                            itemData.configNodes = {}
                        else
                            THUtils.clearTable(itemData.configNodes)
                        end
                        local xmlSubKey = nil
                        for _, configXMLKey in pairs(self.configXMLKeys) do
                            xmlSubKey = configInfo.configurationsKey .. "." .. configXMLKey
                            if THUtils.hasXMLProperty(xmlFile, xmlSubKey) then
                                break
                            end
                        end
                        if xmlSubKey ~= nil then
                            xmlFile:iterate(xmlSubKey .. ".nodeParams.node", function(_, pNodeKey)
                                local nodeId = THUtils.getXMLValue(xmlFile, nil, pNodeKey .. "#node", nil, components, i3dMappings)
                                if THUtils.getNoNil(nodeId, 0) <= 0 then
                                    THUtils.xmlErrorMsg(pNodeKey, nil, THMessage.INVALID_VALUE, "node", nodeId)
                                else
                                    local nodePath = I3DUtil.getNodePath(nodeId)
                                    local shaderParam = THUtils.getXMLValue(xmlFile, nil, pNodeKey .. "#shader")
                                    if nodePath ~= nil then
                                        local nodeData = itemData.configNodes[nodePath]
                                        if nodeData == nil then
                                            nodeData = {
                                                shaderMapping = {}
                                            }
                                            itemData.configNodes[nodePath] = nodeData
                                        end
                                        if shaderParam ~= nil and shaderParam ~= "" then
                                            if not getHasShaderParameter(nodeId, shaderParam) then
                                                THUtils.xmlErrorMsg(pNodeKey, nil, THMessage.INVALID_VALUE, "shader", shaderParam)
                                            else
                                                nodeData.shaderMapping["colorScale0"] = shaderParam
                                            end
                                        end
                                    end
                                end
                            end)
                        end
                        self.configNodes = itemData.configNodes
                    end
                end
            end
        end)
    end
    local function appendFunc(...)
        self.configNodes = nil
        return ...
    end
    return appendFunc(superFunc(item, placeable, ...))
end
function THDesignKit.inj_loadObjectChangeValuesFromXML(self, superFunc, xmlFile, xmlKey, nodeId, object, target, ...)
    THUtils.call(function()
        if THUtils.getIsType(target, Placeable)
            and THUtils.getNoNil(nodeId, 0) > 0
            and xmlKey ~= nil and string.find(xmlKey, "Configurations.", nil, true)
        then
            local xmlFilename = target.configFileName
            local nodePath = I3DUtil.getNodePath(nodeId)
            if nodePath ~= nil
                and xmlFilename ~= nil and xmlFilename ~= ""
            then
                local targetNodes = self.objectChangeNodes[xmlFilename]
                if targetNodes == nil then
                    targetNodes = {}
                    self.objectChangeNodes[xmlFilename] = targetNodes
                end
                local nodeData = targetNodes[nodePath]
                if nodeData == nil then
                    nodeData = {
                        defaultValues = {}
                    }
                    targetNodes[nodePath] = nodeData
                end
            end
        end
    end)
    return superFunc(xmlFile, xmlKey, nodeId, object, target, ...)
end
function THDesignKit.inj_setObjectChange(self, superFunc, object, isActive, target, ...)
    if object ~= nil and target ~= nil then
        THUtils.call(function()
            self:setObjectChangeDefaults(object, true)
        end)
    end
    return superFunc(object, isActive, target, ...)
end
THUtils.call(function()
    local self = THDesignKit.new()
    if self ~= nil then
        _G.g_thDesignKit = self
        source(self.modPath .. "scripts/events/THPlaceableChangeConfigEvent.lua")
        source(self.modPath .. "scripts/gui/THConstructionScreen.lua")
        THUtils.setFunctionHook("TypeManager", "finalizeTypes", false, false, self, THDesignKit.inj_finalizeTypes)
        THUtils.setFunctionHook(g_thGlobalEnv, "setShaderParameter", false, false, self, THDesignKit.inj_setShaderParameter)
        THUtils.setFunctionHook("PlaceableConfigurationItemColor", "registerXMLPaths", false, false, self, THDesignKit.inj_registerPlaceableColorXMLPaths)
        THUtils.setFunctionHook("PlaceableConfigurationItemColor", "onPostLoad", false, false, self, THDesignKit.inj_onPostLoadPlaceableColor)
        THUtils.setFunctionHook("ObjectChangeUtil", "loadValuesFromXML", false, false, self, THDesignKit.inj_loadObjectChangeValuesFromXML)
        THUtils.setFunctionHook("ObjectChangeUtil", "setObjectChange", false, false, self, THDesignKit.inj_setObjectChange)
    end
end)