-- Copyright 2024 by Todd Hundersmarck (ThundR)
-- All Rights Reserved

local thMain = g_thDesignKit
local thInputManager = thMain.inputManager
local thI18n = thMain.i18n
THGuiTopDownCamera = {
    DEFAULTS = {
        DISTANCE_MIN_Z = GuiTopDownCamera.DISTANCE_MIN_Z,
        MOVE_SPEED_FACTOR_NEAR = GuiTopDownCamera.MOVE_SPEED_FACTOR_NEAR,
        MOVE_SPEED_FACTOR_FAR = GuiTopDownCamera.MOVE_SPEED_FACTOR_FAR,
        CAMERA_ZOOM_FACTOR = GuiTopDownCamera.CAMERA_ZOOM_FACTOR,
        CAMERA_ZOOM_FACTOR_MIN = GuiTopDownCamera.CAMERA_ZOOM_FACTOR_MIN,
        COLLISION_MASK = GuiTopDownCamera.COLLISION_MASK
    }
}
THGuiTopDownCamera.activate = function()
    THUtils.call(function()
        GuiTopDownCamera.DISTANCE_MIN_Z = 0
        GuiTopDownCamera.MOVE_SPEED_FACTOR_NEAR = 0.25
        GuiTopDownCamera.MOVE_SPEED_FACTOR_FAR = 32
        GuiTopDownCamera.CAMERA_ZOOM_FACTOR = 0.01
        GuiTopDownCamera.CAMERA_ZOOM_FACTOR_MIN = 0.01
        GuiTopDownCamera.COLLISION_MASK = CollisionFlag.DEFAULT
    end)
end
THGuiTopDownCamera.deactivate = function()
    THUtils.call(function()
        for key, value in pairs(THGuiTopDownCamera.DEFAULTS) do
            GuiTopDownCamera[key] = value
        end
    end)
end
THConstructionScreen = {}
local THConstructionScreen_mt = THUtils.createClass(THConstructionScreen)
THConstructionScreen.ACTION_TEXT = {
    SHOW_CONFIGS = thI18n:getText("input_CONSTRUCTION_SHOW_CONFIGS"),
    PRECISION_MODE = thI18n:getText("input_TH_DESIGN_KIT_PRECISION_MODE"),
    PRECISION_MODE_ON = thI18n:getText("thAction_precisionModeOn"),
    PRECISION_MODE_OFF = thI18n:getText("thAction_precisionModeOff")
}
function THConstructionScreen.getCustomData(target)
    return THUtils.getDataTable(target, THConstructionScreen)
end
THPlaceableEditBrush = {}
function THPlaceableEditBrush.getCustomData(target)
    return THUtils.getDataTable(target, THPlaceableEditBrush)
end
function THConstructionScreen.newData(parent, customMt)
    customMt = customMt or THConstructionScreen_mt
    if THUtils.argIsValid(THUtils.getIsType(parent, ConstructionScreen), "parent", parent)
        and THUtils.argIsValid(type(customMt) == THValueType.TABLE, "customMt", customMt)
    then
        local self = THUtils.createDataTable(parent, THConstructionScreen, customMt)
        if self ~= nil then
            if not self.isInitialized then
                self.isServer = thMain.isServer == true
                self.isClient = thMain.isClient == true
                self.actionEventIds = {}
                self.dialogTexts = {
                    purchaseConfig = thI18n:getText("thDialogText_purchaseConfig")
                }
                self.dialogTitles = {
                    purchaseConfig = thI18n:getText("thDialogTitle_purchaseConfig")
                }
                self.requiredPermissions = {
                    editProperty = {
                        [Farm.PERMISSION.SELL_PLACEABLE] = true
                    }
                }
                self.isPrecisionModeActive = false
                local camera = parent.camera
                if camera ~= nil then
                    local cameraData = THUtils.createDataTable(camera, self)
                    if cameraData ~= nil then
                        cameraData.isEdgeScrollLocked = false
                    end
                end
                if not self.areHooksCreated then
                    THUtils.setFunctionHook(parent, "onClose", false, false, self, THConstructionScreen.inj_onClose)
                    THUtils.setFunctionHook(parent, "onShowConfigs", false, false, self, THConstructionScreen.inj_onShowConfigs)
                    THUtils.setFunctionHook(parent, "update", false, false, self, THConstructionScreen.inj_update)
                    THUtils.setFunctionHook(parent, "mouseEvent", false, false, self, THConstructionScreen.inj_mouseEvent)
                    THUtils.setFunctionHook(parent, "registerBrushActionEvents", false, false, self, THConstructionScreen.inj_registerBrushActionEvents)
                    THUtils.setFunctionHook(parent, "updateBrushActionTexts", false, false, self, THConstructionScreen.inj_updateBrushActionTexts)
                    THUtils.setFunctionHook(parent, "removeBrushActionEvents", false, false, self, THConstructionScreen.inj_removeBrushActionEvents)
                    THUtils.setFunctionHook(parent, "loadCurrentConfiguration", false, false, self, THConstructionScreen.inj_loadCurrentConfiguration)
                    THUtils.setFunctionHook(parent, "processStoreItemConfigurations", false, false, self, THConstructionScreen.inj_processStoreItemConfigurations)
                    if camera ~= nil and camera.setMouseEdgeScrollingActive ~= nil then
                        THUtils.setFunctionHook(camera, "setMouseEdgeScrollingActive", false, false, self, THConstructionScreen.inj_setMouseEdgeScrollingActive)
                    end
                    self.areHooksCreated = true
                end
                self.isInitialized = true
            end
            self.isEnabled = self.isInitialized == true
            return self
        end
    end
end
function THConstructionScreen.actionEditConfigs(self, ...)
    local screen = self.parent
    local allowEdit = false
    if self.isEnabled then
        THUtils.call(function()
            local brush = screen.brush
            if brush ~= nil and brush == screen.selectorBrush then
                local placeable = brush.lastPlaceable
                if placeable ~= nil and placeable.storeItem ~= nil then
                    local editBrush = self.placeableEditBrush
                    if editBrush == nil then
                        editBrush = THPlaceableEditBrush.new(brush)
                        self.placeableEditBrush = editBrush
                    end
                    local brushData = THPlaceableEditBrush.getCustomData(editBrush)
                    if brushData ~= nil and editBrush ~= nil
                        and self:getIsEditAllowed(placeable)
                    then
                        local srcConfigs = placeable.configurations
                        local srcConfigData = placeable.configurationData
                        if srcConfigs ~= nil and next(srcConfigs) ~= nil then
                            local tgtConfigs = THUtils.copyTable(srcConfigs, 1)
                            local tgtConfigData = nil
                            local defaultValues = brushData.placeableDefaults
                            defaultValues.configurations = THUtils.copyTable(srcConfigs, 1)
                            defaultValues.configurationData = nil
                            if srcConfigData ~= nil then
                                tgtConfigData = THUtils.copyTable(srcConfigData, 3)
                                defaultValues.configurationData = THUtils.copyTable(srcConfigData, 3)
                            else
                                tgtConfigData = {}
                                defaultValues.configurationData = {}
                            end
                            editBrush.placeable = placeable
                            editBrush:setStoreItem(placeable.storeItem, tgtConfigs, tgtConfigData)
                            screen.destructMode = false
                            screen:setBrush(editBrush, true)
                            allowEdit = true
                        end
                    end
                end
            end
        end)
    end
    if allowEdit then
        return screen:onShowConfigs(...)
    end
end
function THConstructionScreen.actionTogglePrecisionMode(self, ...)
    local screen = self.parent
    if self.isEnabled and not self.isPrecisionModeActive
        and screen ~= nil and screen.camera ~= nil
        and screen.camera:getIsActive()
    then
        self.isPrecisionModeActive = true
        THGuiTopDownCamera.activate()
    else
        self.isPrecisionModeActive = false
        THGuiTopDownCamera.deactivate()
    end
    self:updateActionEvents()
end
function THConstructionScreen.registerActionEvents(self, noUpdate)
    self:removeActionEvents()
    noUpdate = THUtils.validateArg(not noUpdate or noUpdate == true, "noUpdate", noUpdate, false)
    if self.isEnabled then
        local screen = self.parent
        local brush = screen.brush
        if brush ~= nil then
            local _, eventId = thInputManager:registerActionEvent(InputAction.TH_DESIGN_KIT_PRECISION_MODE, self, self.actionTogglePrecisionMode, false, true, false, true)
            if eventId ~= nil then
                thInputManager:setActionEventTextPriority(eventId, GS_PRIO_VERY_HIGH)
                thInputManager:setActionEventTextVisibility(eventId, false)
                thInputManager:setActionEventActive(eventId, false)
            end
            self.actionEventIds.precisionMode = eventId
            if brush == screen.selectorBrush then
                _, eventId = thInputManager:registerActionEvent(InputAction.TH_DESIGN_KIT_CUSTOMIZE, self, self.actionEditConfigs, false, true, false, true)
                if eventId ~= nil then
                    thInputManager:setActionEventText(eventId, THConstructionScreen.ACTION_TEXT.SHOW_CONFIGS)
                    thInputManager:setActionEventTextPriority(eventId, GS_PRIO_VERY_HIGH)
                    thInputManager:setActionEventTextVisibility(eventId, false)
                    thInputManager:setActionEventActive(eventId, false)
                end
                self.actionEventIds.editConfigs = eventId
            end
            if noUpdate ~= true then
                self:updateActionEvents()
            end
        end
    end
end
function THConstructionScreen.removeActionEvents(self)
    for eventName, eventId in pairs(self.actionEventIds) do
        thInputManager:removeActionEvent(eventId)
        self.actionEventIds[eventName] = nil
    end
end
function THConstructionScreen.updateActionEvents(self)
    local mission = g_currentMission
    local screen = self.parent
    if self.isEnabled then
        local brush = screen.brush
        for _, eventId in pairs(self.actionEventIds) do
            local isEventActive = false
            if eventId == self.actionEventIds.editConfigs then
                if mission ~= nil
                    and brush ~= nil and brush == screen.selectorBrush
                    and brush.lastPlaceable ~= nil
                    and brush.lastPlaceable.configurations ~= nil
                    and next(brush.lastPlaceable.configurations) ~= nil
                    and self:getIsEditAllowed(brush.lastPlaceable)
                then
                    isEventActive = true
                end
            elseif eventId == self.actionEventIds.precisionMode then
                if self.isPrecisionModeActive then
                    thInputManager:setActionEventText(eventId, THConstructionScreen.ACTION_TEXT.PRECISION_MODE_OFF)
                else
                    thInputManager:setActionEventText(eventId, THConstructionScreen.ACTION_TEXT.PRECISION_MODE_ON)
                end
                isEventActive = true
            end
            thInputManager:setActionEventTextVisibility(eventId, isEventActive)
            thInputManager:setActionEventActive(eventId, isEventActive)
        end
    else
        for _, eventId in pairs(self.actionEventIds) do
            thInputManager:setActionEventTextVisibility(eventId, false)
            thInputManager:setActionEventActive(eventId, false)
        end
    end
end
function THConstructionScreen.getIsEditAllowed(self, object, farmId)
    local mission = g_currentMission
    if THUtils.argIsValid(object == nil or THUtils.getIsType(object, Object), "object", object)
        and self.isEnabled and mission ~= nil
    then
        if THUtils.getIsMasterUser() then
            return true
        end
        local allowAccess = true
        for permissionId in pairs(self.requiredPermissions.editProperty) do
            if not mission:getHasPlayerPermission(permissionId) then
                allowAccess = false
                break
            end
        end
        if allowAccess and object ~= nil then
            farmId = farmId or mission:getFarmId()
            if not mission.accessHandler:canFarmAccess(farmId, object) then
                allowAccess = false
            end
        end
        return allowAccess
    end
    return false
end
function THConstructionScreen.setCurrentConfiguration(self, configs, configData, isPurchased, purchasePrice)
    isPurchased = THUtils.validateArg(not isPurchased or isPurchased == true, "isPurchased", isPurchased, false)
    local mission = g_currentMission
    local success = false
    if THUtils.argIsValid(configs == nil or type(configs) == THValueType.TABLE, "configs", configs)
        and THUtils.argIsValid(configData == nil or type(configData) == THValueType.TABLE, "configData", configData)
        and THUtils.argIsValid(purchasePrice == nil or (type(purchasePrice) == THValueType.NUMBER and purchasePrice >= 0), "purchasePrice", purchasePrice)
        and self.isEnabled and mission ~= nil
    then
        local screen = self.parent
        local brushData, brush = THPlaceableEditBrush.getCustomData(screen.brush)
        if brushData ~= nil and brush == self.placeableEditBrush then
            local placeable = brush.placeable
            if placeable ~= nil then
                configs = configs or placeable.configurations
                configData = configData or placeable.configurationData
                if configs ~= nil and configData ~= nil
                    and self:getIsEditAllowed(placeable)
                then
                    local playerFarmId = mission:getFarmId()
                    local _, isProcessed = nil, false
                    local failMsgText = nil
                    if isPurchased then
                        if purchasePrice == nil then
                            _, purchasePrice = mission.economyManager:getBuyPrice(placeable.storeItem, configs)
                            purchasePrice = math.max(0, purchasePrice or 0)
                        end
                    else
                        purchasePrice = 0
                    end
                    local function processChanges()
                        if thMain:updateConfigurations(placeable, configs, configData, nil, isPurchased, purchasePrice, playerFarmId) then
                            brush:setStoreItem(placeable.storeItem, configs, configData)
                            isProcessed = true
                        end
                    end
                    if isPurchased then
                        local currentMoney = mission:getMoney(playerFarmId) or 0
                        if purchasePrice <= 0 or (currentMoney - purchasePrice >= 0) then
                            processChanges()
                        else
                            failMsgText = thI18n:getText("thMessage_notEnoughMoney")
                        end
                    else
                        processChanges()
                    end
                    if not isProcessed then
                        local defaultValues = brushData.placeableDefaults
                        configs = defaultValues.configurations or placeable.configurations
                        configData = defaultValues.configurationData or placeable.configurationData
                        isPurchased = true
                        purchasePrice = 0
                        processChanges()
                        if failMsgText ~= nil then
                            mission:showBlinkingWarning(failMsgText, 2000)
                        end
                    end
                    success = isProcessed == true
                end
            end
        end
    end
    return success
end
function THConstructionScreen.inj_onClose(self, superFunc, screen, ...)
    THUtils.call(function()
        self.isPrecisionModeActive = false
        THGuiTopDownCamera.deactivate()
    end)
    return superFunc(screen, ...)
end
function THConstructionScreen.inj_onShowConfigs(self, superFunc, screen, ...)
    local function appendFunc(...)
        THUtils.call(function()
            if screen.configsBox:getIsVisible() then
                self:registerActionEvents()
            end
        end)
        return ...
    end
    return appendFunc(superFunc(screen, ...))
end
function THConstructionScreen.inj_update(self, superFunc, screen, dt, ...)
    local function appendFunc(...)
        THUtils.call(function()
            local isConfigsBoxVisible = screen.configsBox:getIsVisible()
            if not isConfigsBoxVisible then
                local brushData, brush = THPlaceableEditBrush.getCustomData(screen.brush)
                local function deactivateBrush()
                    brush.placeable = nil
                    screen:setBrush(brushData.parent, true)
                end
                if brushData ~= nil and brush ~= nil and brush == self.placeableEditBrush then
                    if brush.placeable ~= nil then
                        local function restoreDefaults()
                            local defaultValues = brushData.placeableDefaults
                            self:setCurrentConfiguration(defaultValues.configurations, defaultValues.configurationData, true, 0)
                        end
                        YesNoDialog.show(function(pIsYes)
                            THUtils.call(function()
                                if pIsYes then
                                    self:setCurrentConfiguration(nil, nil, true)
                                else
                                    restoreDefaults()
                                end
                                deactivateBrush()
                            end)
                        end, nil, self.dialogTexts.purchaseConfig, self.dialogTitles.purchaseConfig)
                    else
                        deactivateBrush()
                    end
                end
            end
        end)
        return ...
    end
    return appendFunc(superFunc(screen, dt, ...))
end
function THConstructionScreen.inj_mouseEvent(self, superFunc, screen, ...)
    local function appendFunc(...)
        THUtils.call(function()
            if screen.camera ~= nil and screen.camera.setMouseEdgeScrollingActive ~= nil then
                local cameraData = THUtils.getDataTable(screen.camera, self)
                if cameraData ~= nil then
                    local isConfigsBoxVisible = screen.configsBox:getIsVisible()
                    if isConfigsBoxVisible and self.isPrecisionModeActive then
                        if not cameraData.isEdgeScrollLocked then
                            cameraData.lastIsEdgeScrollActive = screen.camera.isMouseEdgeScrollingActive
                            screen.camera:setMouseEdgeScrollingActive(false)
                            cameraData.isEdgeScrollLocked = true
                        end
                    else
                        if cameraData.isEdgeScrollLocked then
                            cameraData.isEdgeScrollLocked = false
                            if cameraData.lastIsEdgeScrollActive ~= nil then
                                screen.camera:setMouseEdgeScrollingActive(cameraData.lastIsEdgeScrollActive)
                                cameraData.lastIsEdgeScrollActive = nil
                            end
                        end
                    end
                end
            end
        end)
        return ...
    end
    return appendFunc(superFunc(screen, ...))
end
function THConstructionScreen.inj_registerBrushActionEvents(self, superFunc, screen, ...)
    local function appendFunc(...)
        THUtils.call(function()
            self:registerActionEvents()
        end)
        return ...
    end
    return appendFunc(superFunc(screen, ...))
end
function THConstructionScreen.inj_updateBrushActionTexts(self, superFunc, screen, ...)
    local function appendFunc(...)
        THUtils.call(function()
            self:updateActionEvents()
        end)
        return ...
    end
    return appendFunc(superFunc(screen, ...))
end
function THConstructionScreen.inj_removeBrushActionEvents(self, superFunc, screen, ...)
    local function appendFunc(...)
        THUtils.call(function()
            self:removeActionEvents()
        end)
        return ...
    end
    return appendFunc(superFunc(screen, ...))
end
function THConstructionScreen.inj_loadCurrentConfiguration(self, superFunc, screen, storeItem, ...)
    local prependSuccess = false
    THUtils.call(function()
        local brushData, brush = THPlaceableEditBrush.getCustomData(screen.brush)
        if brushData ~= nil and brush ~= nil and brush == self.placeableEditBrush then
            if self.isEnabled then
                if screen.configurations ~= nil and screen.configurationData ~= nil then
                    self:setCurrentConfiguration(screen.configurations, screen.configurationData)
                end
            end
            prependSuccess = true
        end
    end)
    if not prependSuccess then
        return superFunc(screen, storeItem, ...)
    end
end
function THConstructionScreen.inj_processStoreItemConfigurations(self, superFunc, screen, storeItem, ...)
    THUtils.call(function()
        local brushData, brush = THPlaceableEditBrush.getCustomData(screen.brush)
        if brushData ~= nil and brush ~= nil and brush == self.placeableEditBrush then
            if brush.configurationData ~= nil then
                screen.configurationData = THUtils.copyTable(brush.configurationData, 3)
            end
        end
    end)
    return superFunc(screen, storeItem, ...)
end
function THConstructionScreen.inj_setMouseEdgeScrollingActive(self, superFunc, camera, isActive, ...)
    local allowUpdate = true
    THUtils.call(function()
        local cameraData = THUtils.getDataTable(camera, self)
        if cameraData ~= nil then
            if cameraData.isEdgeScrollLocked then
                if isActive ~= nil then
                    cameraData.lastIsEdgeScrollActive = isActive
                end
                allowUpdate = false
            else
                cameraData.lastIsEdgeScrollActive = nil
            end
        end
    end)
    if allowUpdate then
        return superFunc(camera, isActive, ...)
    end
end
function THConstructionScreen.inj_setBrush(superFunc, screen, brush, ...)
    local self = THConstructionScreen.getCustomData(screen)
    THUtils.call(function()
        if self == nil then
            self = THConstructionScreen.newData(screen)
        end
    end)
    return superFunc(screen, brush, ...)
end
THUtils.setFunctionHook("ConstructionScreen", "setBrush", false, false, nil, THConstructionScreen.inj_setBrush)
function THPlaceableEditBrush.new(parent)
    if THUtils.argIsValid(THUtils.getIsType(parent, ConstructionBrushSelect), "parent", parent) then
        local self = setmetatable({}, { __index = parent })
        local customData = THUtils.createDataTable(self, THPlaceableEditBrush)
        customData.parent = parent
        customData.isEditBrush = true
        customData.placeableDefaults = {}
        return self
    end
end