-- Copyright 2024-2025 by Todd Hundersmarck (ThundR) 
-- All Rights Reserved

local thModName = g_currentModName
local thModPath = g_currentModDirectory
THModManager = {}
local THModManager_mt = Class(THModManager)
THModManager.MOD = {}
function THModManager.new(customMt)
    customMt = customMt or THModManager_mt
    if THUtils.argIsValid(type(customMt) == THValueType.TABLE, "customMt", customMt) then
        local mpLoadingScreen = g_mpLoadingScreen
        local missionDynamicInfo = mpLoadingScreen.missionDynamicInfo
        local loadedMods = missionDynamicInfo.mods
        local self = setmetatable({}, customMt)
        self.modName = thModName
        self.modPath = thModPath
        self.loadedMods = {
            byId = {},
            byIndex = {}
        }
        self.modList = {}
        if loadedMods ~= nil then
            for key, value in pairs(loadedMods) do
                self.modList[key] = value
            end
        end
        self:initModData()
        return self
    end
end
function THModManager.initModData(self)
    for _, modInfo in ipairs(self.modList) do
        local modName = THUtils.getNoNil(modInfo.modName, "")
        if modName ~= "" then
            local modEnv = THUtils.getModEnvironment(modName)
            if modEnv == nil then
                THUtils.errorMsg(false, "Can not find mod environment: %s", modName)
            else
                local modData = {
                    id = modName:upper(),
                    name = modName,
                    info = modInfo,
                    path = modInfo.modDir,
                    file = modInfo.modFile,
                    index = #self.loadedMods.byIndex + 1,
                    title = modInfo.title,
                    descVersion = modInfo.modDescVersion,
                    environment = modEnv
                }
                local modVersion = modInfo.version
                if type(modVersion) == THValueType.STRING then
                    local versionData = THUtils.splitString(modVersion, ".", true)
                    if versionData == nil then
                        versionData = {}
                    end
                    modData.version = {
                        text = modVersion,
                        major = THUtils.toNumber(versionData[1], true) or 0,
                        minor = THUtils.toNumber(versionData[2], true) or 0,
                        revision = THUtils.toNumber(versionData[3], true) or 0,
                        build = THUtils.toNumber(versionData[4], true) or 0
                    }
                else
                    modData.version = {
                        text = "0.0.0.0",
                        major = 0,
                        minor = 0,
                        revision = 0,
                        build = 0
                    }
                end
                self.loadedMods.byId[modData.id] = modData
                self.loadedMods.byIndex[modData.index] = modData
                if THGameVersion == 25 then
                    if modData.id == "FS25_PRECISIONFARMING" then
                        local otherModEnv = modData.environment
                        if otherModEnv ~= nil then
                            modData.target = otherModEnv.g_precisionFarming
                            if modData.target ~= nil then
                                THModManager.MOD.PRECISION_FARMING = modData.index
                            end
                        end
                    end
                end
            end
        end
    end
end
function THModManager.getLoadedMods(self)
    return self.loadedMods.byIndex, #self.loadedMods.byIndex
end
function THModManager.getLoadedMod(self, modId, verbose)
    verbose = THUtils.validateArg(not verbose or verbose == true, "verbose", verbose, true)
    local modIdType = type(modId)
    local modData = nil
    if modId ~= nil then
        if modIdType == THValueType.STRING then
            modData = self.loadedMods.byId[modId:upper()]
        elseif modIdType == THValueType.NUMBER then
            modData = self.loadedMods.byIndex[modId]
        else
            verbose = true
        end
    end
    if modData == nil and verbose == true then
        THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "modId", modId)
    end
    return modData
end
function THModManager.getLoadedModByFilename(self, filename)
    if THUtils.argIsValid(filename == nil or type(filename) == THValueType.STRING, "filename", filename) then
        local absFilename = THUtils.getFilename(filename)
        if absFilename ~= nil and absFilename ~= "" and absFilename ~= "/" then
            local fileParts = THUtils.splitString(absFilename, "/")
            if fileParts ~= nil and next(fileParts) ~= nil then
                for partIndex = 1, #fileParts do
                    local partName = string.gsub(fileParts[partIndex], "%$", "")
                    if partName:upper() ~= "DATA" then
                        local modData = self:getLoadedMod(partName)
                        if modData ~= nil then
                            return modData
                        end
                    end
                end
            end
        end
    end
end
function THModManager.getLoadedModByClassName(self, className, modName)
    local modData = nil
    if modName ~= nil then
        modData = self:getLoadedMod(modName)
        if modData == nil then
            return
        end
    end
    if modData == nil then
        local loadedModsArray, numLoadedMods = self:getLoadedMods()
        if numLoadedMods > 0 then
            for modIndex = 1, numLoadedMods do
                modData = loadedModsArray[modIndex]
                local modClass = modData.environment[className]
                if type(modClass) == THValueType.TABLE then
                    return modData, modClass
                end
            end
        end
    else
        local modClass = modData.environment[className]
        if type(modClass) == THValueType.TABLE then
            return modData, modClass
        end
    end
end
function THModManager.getModVersion(self, modName)
    local modData = self:getLoadedMod(modName)
    local modVersionText = "0.0.0.0"
    local modVersionData = nil
    if modData ~= nil then
        modVersionData = modData.version
        if modVersionData ~= nil then
            modVersionText = modVersionData.text or modVersionText
        end
    end
    return modVersionText, modVersionData
end
local thModManager = THModManager.new()
if thModManager ~= nil then
    _G.g_thModManager = thModManager
end